/**
 * List component.
 *
 * @author: Vaclav Hradec <vaclav@mihybrid.com>
 *
 */

'use strict';

import { addClass, removeClass, hasClass } from "mihbbtv/lib/util/classes";
import List from "mihbbtv/lib/components/list";
import KeyHandler from "mihbbtv/lib/hbbtv/keyHandler";
import Resizer from "mihbbtv/lib/util/resizer";
import Sanitize from "mihbbtv/lib/util/sanitize";
import Image from "mihbbtv/lib/components/image";

import Item from "../models/item";
import Config from "../config";

class AppList extends List {

  constructor(cfg, parent) {
    // bypass first and last to be not presented
    if (cfg.pagination && cfg.pagination.links) {
      delete cfg.pagination.links.first;
      delete cfg.pagination.links.last;
    }

    super(cfg, parent);

    // console.log(cfg);

    this.isFlattened = false;
  }

  enterItem(item, app, scene) {
    const elm = this.elms[this.getActiveIndex()];

    // console.log(item);

    if ('pagination' === item.type) {
      return app.setScene('detail', null, {
        componentIndex: ('first' === item.action || ('prev' === item.action && item.isOneAfterStart)) ? 0 : 2,
        pagination: {
          url: item.url,
      }});
    } else if ('serial' === item.type) {
      if (this.isInSubmenu) {
        return Item.enterItem(item.items[this.subActiveIndex], app, scene, this.subElms[this.getActiveIndex()][this.subActiveIndex]);
      } else {
        this.enterSubmenu();
      }
    } else {
      return Item.enterItem(item, app, scene, elm);
    }
  }

  paginationMarkup(item) {
    let content = '';

    switch (item.action) {
      case 'first':
      content = '|&lt; První';
      break;
      case 'last':
      content = 'Poslední &gt;|';
      break;
      case 'next':
      content = 'Další &gt;';
      break;
      case 'prev':
      content = '&lt; Předchozí';
      break;
    }

    return content;
  }

  itemMarkup(item, index) {
    return (('audio' === item.type) ? (
      '<div class="player-btn"></div>'
    ) :
    '') +
    '<div class="col duration">' + item.duration + '</div>' +
    '<div class="col date">' + item.date + '</div>' +
    '<div class="col time">' + item.time + '</div>' +
    '<div class="col headline">' + Sanitize.html(item.headline, 75) + '</div>'
    ;
  }

  groupMarkup(item, index) {
    return '<div class="col img">' + ((item.img) ?
    Image.markup({
      src: (Config.resizer && Config.resizer.enabled) ? Resizer.url(item.img, 130, 130) : item.img,
      rounded: {
        tl: true,
        tr: true,
        bl: true,
        br: true,
      }
    }) : ''
    ) + '</div>' +
    ((item.duration) ? ('<div class="col duration">' + item.duration + '</div>') : '') +
    ((item.date) ? ('<div class="col date">' + item.date + '</div>') : '') +
    ((item.time) ? ('<div class="col time">' + item.time + '</div>') : '') +
    '<div class="col headline">' + Sanitize.html(item.headline, 75) + '</div>' +
    '<div class="col desc">' + Sanitize.html(item.text, 240) + '</div>' +
    (('serial' === item.type) ? (
      '<div class="player-btn serial">' + item.items.length + '</div>'
    ) : (('audio' === item.type) ? '<div class="player-btn"></div>' : ''))
    ;
  }

  
  scroll(toIndex) {
    const actvIndex = (this.isInSubmenu) ? this.subActiveIndex : this.getActiveIndex();

    if (this.parentScroller) {
      // parent scrolling

      // console.log(this.isInSubmenu, toIndex, actvIndex, this.items, this.subElms);

      const submenuItemHeight = (this.isInSubmenu) ? this.subElms[this.getActiveIndex()][this.subActiveIndex].offsetHeight : this.itemHeight;
      // console.log(`parent scroll`, toIndex, submenuItemHeight);

      // simplification - dont scroll on pagination items
      if (toIndex > 0 && this.items[toIndex] && 'pagination' === this.items[toIndex].type) {
        return false;
      }

      const fromBottom = actvIndex > toIndex || (!actvIndex && toIndex === this.items.length - 1);

      // console.log(`fromBottom`, fromBottom, toIndex, this.items.length - 1, this.getEndShift());

      if (! this.isInSubmenu && (toIndex === 0 || toIndex <= this.getStartShift())) {
        // console.log('list scroll to start');
        this.parentScroller.startY();
      } else
      if (! this.isInSubmenu && fromBottom && (toIndex === this.items.length - 1 || toIndex >= this.items.length - 1 - this.getEndShift())) {
        // dont scroll if go from bottom and go to last item
        return false;
      } else
      if (toIndex < actvIndex) {
        this.parentScroller.directMoveYDelta(-1 * submenuItemHeight);
      } else if (toIndex > actvIndex) {
        this.parentScroller.directMoveYDelta(submenuItemHeight);
      }

      return true;
    }

    if (! this.scroller) {
      return false;
    }

    // console.log(`scroll`, toIndex, actvIndex);

    if (! toIndex) {
      // to start
      this.scroller.startY();
    } else if (toIndex === this.totalCnt) {
      // to end
      this.scroller.endY();
    } else if (this.needToScroll(toIndex)) {
      this.scroller.setY(toIndex / this.itemsInView());
    }
  }

  keyHandlerButtons(kc, app, scene) {

    if ((kc === KeyHandler.keycodes.VK_PLAY) || (kc === KeyHandler.keycodes.VK_PLAY_PAUSE)) {
      this.enterItem(this.items[this.getActiveIndex()], app, scene);

      return true;
    }

    return super.keyHandlerButtons(kc, app, scene);
  }

}

export default AppList;
