package sk.kosice.konto.kknotificationservice.repository.recipient;

import static sk.kosice.konto.kkmessageservice.repository.model.Tables.RECIPIENT;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.SUBSCRIPTION;
import static sk.kosice.konto.kknotificationservice.domain.recipient.error.RecipientErrorCode.RECIPIENT_DOES_NOT_EXIST;
import static sk.kosice.konto.kknotificationservice.repository.recipient.mapper.JooqRecipientRepositoryMapper.recipientRecordMapper;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import org.jooq.DSLContext;
import org.jooq.TableField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import sk.kosice.konto.kknotificationservice.domain.common.error.BusinessException;
import sk.kosice.konto.kknotificationservice.domain.recipient.entity.RecipientEntity;
import sk.kosice.konto.kknotificationservice.repository.JooqRepository;
import sk.kosice.konto.kknotificationservice.repository.recipient.error.constraint.RecipientConstraintErrorCode;

@Repository
public class JooqRecipientRepository extends JooqRepository implements RecipientRepositoryAdapter {

  private Map<TableField<?, ?>, Object> insertValues(RecipientEntity data) {
    final Map<TableField<?, ?>, Object> map = new HashMap<>();
    map.put(RECIPIENT.KID, data.kid());
    map.put(RECIPIENT.EMAIL, data.email());
    return map;
  }

  @Autowired
  protected JooqRecipientRepository(DSLContext dslContext) {
    super(dslContext);
  }

  @Override
  public void merge(RecipientEntity data) throws BusinessException {
    log.info("Applying {}", data);

    handleMergeResult(
        () ->
            dslContext
                .insertInto(RECIPIENT)
                .set(insertValues(data))
                .onConflict(RECIPIENT.KID)
                .doUpdate()
                .set(RECIPIENT.EMAIL, data.email())
                .execute(),
        data,
        RecipientConstraintErrorCode.values());
  }

  @Override
  public RecipientEntity findOne(UUID kid) throws BusinessException {
    log.info("Applying {}", kid);

    return handleFindOneResult(
        () ->
            dslContext
                .selectFrom(RECIPIENT)
                .where(RECIPIENT.KID.eq(kid))
                .fetch(recipientRecordMapper()),
        kid,
        RecipientConstraintErrorCode.values());
  }

  @Override
  public List<RecipientEntity> list() {
    return dslContext.selectFrom(RECIPIENT).fetch(recipientRecordMapper());
  }

  @Override
  public List<RecipientEntity> listByTopic(UUID topicId) {
    return dslContext
        .select(RECIPIENT)
        .from(RECIPIENT)
        .join(SUBSCRIPTION)
        .on(RECIPIENT.KID.eq(SUBSCRIPTION.RECIPIENT_KID))
        .where(SUBSCRIPTION.TOPIC_ID.eq(topicId).and(SUBSCRIPTION.IS_EMAIL_ENABLED.eq(true)))
        .fetch(r -> recipientRecordMapper().map(r.value1()));
  }

  @Override
  protected void createEntityDoesNotExistError(Object id) {
    throw RECIPIENT_DOES_NOT_EXIST.createError(id).convertToException();
  }
}
