package sk.kosice.konto.kknotificationservice.restapi.config;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.info.Info;
import io.swagger.v3.oas.models.info.License;
import io.swagger.v3.oas.models.security.SecurityRequirement;
import io.swagger.v3.oas.models.security.SecurityScheme;
import java.util.Arrays;
import org.springdoc.core.customizers.OpenApiCustomizer;
import org.springdoc.core.models.GroupedOpenApi;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import sk.kosice.konto.kknotificationservice.restapi.controller.RecipientController;

@Configuration
public class ModuleWebApiConfig {

  public static final String BEARER_AUTH = "bearerAuth";

  @Bean
  ObjectMapper objectMapper(Jackson2ObjectMapperBuilder builder) {
    final ObjectMapper mapper =
        builder.featuresToEnable(SerializationFeature.WRITE_DATE_KEYS_AS_TIMESTAMPS).build();
    return mapper.registerModules(new JavaTimeModule(), new Jdk8Module()).findAndRegisterModules();
  }

  @Bean
  public OpenAPI openAPI(@Value("${application.rest-api.version:unknown}") String appVersion) {
    return new OpenAPI()
        .info(
            new Info()
                .title("KK-Notification Component")
                .version(appVersion)
                .description("Open API documentation for kknotification microservice.")
                .termsOfService("TODO")
                .license(
                    new License().name("MIT License").url("https://opensource.org/license/mit")));
  }

  @Bean
  public GroupedOpenApi groupedAppIntegrationApi(
      @Qualifier("apiIntegrationAuthApiCustomizer")
          OpenApiCustomizer apiIntegrationAuthApiCustomizer) {
    return GroupedOpenApi.builder()
        .group("App integration REST API v1")
        .pathsToMatch(prepareURIsForApiGroup(RecipientController.RECIPIENT_URI))
        .addOpenApiCustomizer(apiIntegrationAuthApiCustomizer)
        .build();
  }

  @Bean(name = "apiIntegrationAuthApiCustomizer")
  public OpenApiCustomizer apiIntegrationAuthApiCustomizer(
      OAuthPropertiesPort oAuthPropertiesPort) {
    return openApi ->
        openApi
            .addSecurityItem(new SecurityRequirement().addList("openId"))
            .components(
                openApi
                    .getComponents()
                    .addSecuritySchemes(
                        BEARER_AUTH,
                        new SecurityScheme()
                            .description(
                                "OpenID by auto-configuration from /.well-known/openid-configuration")
                            .type(SecurityScheme.Type.OPENIDCONNECT)
                            .openIdConnectUrl(
                                oAuthPropertiesPort.ocAppIntegrationIssuerLocation()
                                    + "/.well-known/openid-configuration")
                            .in(SecurityScheme.In.HEADER)));
  }

  private String[] prepareURIsForApiGroup(String... paths) {
    return Arrays.stream(paths)
        .map(path -> "/" + path.replaceAll("\\{\\w*\\}", "*"))
        .toArray(String[]::new);
  }
}
