import NextAuth from 'next-auth';
import MicrosoftEntraId from 'next-auth/providers/microsoft-entra-id';
import { isExpired, refreshToken } from './utils/auth';
import { IUser, Permission } from './utils/api';
import { getOrganization } from './utils/auth/server';
import jwt from 'jsonwebtoken';

export const { handlers, auth, signIn, signOut } = NextAuth({
  providers: [
    MicrosoftEntraId({
      id: 'microsoft-entra-id',
      issuer: process.env.AZURE_AD_ISSUER,
      clientId: process.env.AZURE_AD_CLIENT_ID,
      clientSecret: process.env.AZURE_AD_CLIENT_SECRET,
      authorization: {
        url: process.env.AZURE_AD_AUTH_URL,
        response_type: 'code',
        params: {
          scope: process.env.AZURE_AD_SCOPE
        }
      },
      token: {
        url: process.env.AZURE_AD_TOKEN_URL,
        response_type: 'token id_token',
        params: {
          scope: process.env.AZURE_AD_SCOPE
        }
      },

      profile: (profile) => ({
        groups: profile.groups,
        email: ''
      })
    })
  ],
  callbacks: {
    jwt: async ({ token, account, trigger, session }) => {
      const typedToken = token as any;

      if (trigger === 'update') {
        const { organization, permissions } = session.permission;
        typedToken.user.organization = organization;
        typedToken.user.permissions = permissions;
        return typedToken;
      }

      if (account) {
        if (account.id_token && account.access_token) {
          const accessToken = jwt.decode(account.access_token) as any;

          let permission: Permission | null = null;
          const { data, error } = await getOrganization(accessToken.oid, account.access_token);

          if (error) {
            typedToken.error = 'OrganizationError';
          } else if (data.items.length === 0) {
            typedToken.error = 'NoOrganizationError';
          } else {
            permission = data.items[0];
          }

          typedToken.user = {
            oid: accessToken.oid,
            name: accessToken.name,
            email: accessToken.preferred_username,
            organization: permission?.organization,
            permissions: permission?.permissions || []
          } as IUser;

          typedToken.accessToken = account.access_token;
          typedToken.refreshToken = account.refresh_token;
          typedToken.id_token = account.id_token;
          typedToken.expiresAt = account.expires_at;
        } else {
          typedToken.error = 'MissingTokenError';
        }
      }

      if (isExpired(typedToken.expiresAt)) {
        try {
          const refreshedToken = await refreshToken(typedToken);
          return refreshedToken;
        } catch (_) {
          typedToken.error = 'RefreshTokenError';
        }
      }

      return typedToken;
    },

    session: async ({ session, token }) => {
      const sameSession = session as any;
      const sameToken = token as any;

      if (sameToken.user) {
        sameSession.user = sameToken.user;
      }

      sameSession.idToken = sameToken.idToken;
      sameSession.accessToken = sameToken.accessToken;
      sameSession.refreshToken = sameToken.refreshToken;
      sameSession.expiresAt = sameToken.expiresAt;
      sameSession.error = sameToken.error;

      return session;
    }
  },
  pages: {
    error: '/error',
    signIn: '/signin'
  },
  session: { strategy: 'jwt' },
  secret: process.env.NEXTAUTH_SECRET,
  trustHost: true
});
