import { useLocale, useTranslations } from 'next-intl';
import React, { useCallback, useMemo, useState } from 'react';
import {
  Empty,
  QueryHandler,
  TableList,
  TableListItem,
  TableListValue,
  TableSection,
  Pagination,
  Error
} from '@/components';
import { Loader, Table, TableRow, TableRowValue } from '@/lib/idsk';
import { useGetProperties } from '@/hooks';
import { Construction, Flat, Parcel, Properties as PropertyDto } from '@/types';
import { date } from '@/utils';

const PAGE_SIZE = 10;

type SimpleProperty = {
  id: string;
  address: string;
  type: string;
  size: number;
  date: string;
};

interface PropertyProps {
  session: any;
  view: 'table' | 'list';
}

interface PropertyEntryProps {
  property: SimpleProperty;
  view: 'list' | 'table';
}

interface PropertiesListProps {
  view: 'table' | 'list';
  propertyDto: PropertyDto;
  page: number;
  onPageChange: (page: number) => void;
}

const PropertyEntry: React.FC<PropertyEntryProps> = ({ property, view }) => {
  const t = useTranslations('my_data');

  if (view === 'table') {
    return (
      <TableRow className="tb2:flex w-full">
        <TableRowValue className="tb2:w-[40%]">{property.address}</TableRowValue>
        <TableRowValue className="tb2:w-[25%]">{property.type}</TableRowValue>
        <TableRowValue className="tb2:w-[20%]">{property.size}</TableRowValue>
        <TableRowValue className="tb2:w-[15%]" align="right">
          {property.date}
        </TableRowValue>
      </TableRow>
    );
  }

  return (
    <TableListItem>
      <TableListValue label={t('property_address')}>{property.address}</TableListValue>
      <TableListValue label={t('property_type')}>{property.type}</TableListValue>
      <TableListValue label={t('property_size')}>{property.size}</TableListValue>
      <TableListValue label={t('date_of_registration')}>{property.date}</TableListValue>
    </TableListItem>
  );
};

const PropertiesList: React.FC<PropertiesListProps> = ({ view, propertyDto, page, onPageChange }) => {
  const t = useTranslations('my_data');
  const locale = useLocale();

  const mapToSimpleProperty = useCallback(
    (property: Flat | Parcel | Construction): SimpleProperty => {
      let type = t(property.type);
      if (type.startsWith('my_data.')) {
        type = t('unknown_type');
      }

      const prop = property as any;
      const size = prop.flatArea || prop.parcelArea || 0;

      let address = '';
      if (prop.type === 'c_parcel') {
        address += t('parcel_number') + ' ' + prop.parcelNumber;
      } else {
        address += prop.street;
      }
      if (prop.constructionRegistryNumber) {
        address += ` ${prop.constructionRegistryNumber}`;
      }
      if (prop.entaranceNumber) {
        address += `/${prop.entaranceNumber}`;
      }

      address += `, ${prop.municipality}`;

      return {
        id: property.prid,
        address,
        date: date(property.created, locale),
        size,
        type
      };
    },
    [locale, t]
  );

  const properties: SimpleProperty[] = useMemo(() => {
    const items = propertyDto.properties;

    return items.reduce((acc: SimpleProperty[], current) => {
      const flats = current.flats.map(mapToSimpleProperty);
      const constructions = current.constructions.map(mapToSimpleProperty);
      const parcels = current.parcels.map(mapToSimpleProperty);
      return [...acc, ...flats, ...constructions, ...parcels];
    }, []);
  }, [mapToSimpleProperty, propertyDto.properties]);

  if (properties.length === 0) {
    return <Empty title={t('empty_properties')} asSection />;
  }

  return (
    <>
      {view === 'table' ? (
        <Table
          className="properties-table"
          headRowBordered
          headRow={
            <>
              <TableRowValue className="tb2:w-[40%]">{t('property_address')}</TableRowValue>
              <TableRowValue className="tb2:w-[25%]">{t('property_type')}</TableRowValue>
              <TableRowValue className="tb2:w-[20%]">{t('property_size')}</TableRowValue>
              <TableRowValue className="tb2:w-[15%]" align="right">
                {t('date_of_registration')}
              </TableRowValue>
            </>
          }
        >
          {properties.map((property) => (
            <PropertyEntry key={property.id} property={property} view={view} />
          ))}
        </Table>
      ) : (
        <TableList>
          {properties.map((property) => (
            <PropertyEntry key={property.id} property={property} view={view} />
          ))}
        </TableList>
      )}
      {propertyDto.pagination.size > PAGE_SIZE && (
        <Pagination
          className="ml-auto"
          page={page}
          pageSize={PAGE_SIZE}
          totalCount={propertyDto.pagination.size}
          onPageChange={(selected) => onPageChange(selected)}
        />
      )}
    </>
  );
};

const Property: React.FC<PropertyProps> = ({ session, view }) => {
  const [page, setPage] = useState(1);

  const t = useTranslations('my_data');

  const { data } = session;

  const userId = data?.user.id || '';
  const query = useGetProperties(userId, page, PAGE_SIZE);

  return (
    <TableSection title={t('properties')} description={t('properties_description')}>
      <QueryHandler
        query={query}
        loading={
          <div className="px-6 py-12 flex items-center justify-center w-full">
            <Loader />
          </div>
        }
        error={<Error err={query.error} />}
      >
        {(propertyDto: PropertyDto) => (
          <PropertiesList
            view={view}
            propertyDto={propertyDto}
            page={page}
            onPageChange={(selected) => setPage(selected)}
          />
        )}
      </QueryHandler>
    </TableSection>
  );
};

export default Property;
