'use client';

import axios from 'axios';
import React, { useEffect } from 'react';
import { useQuery } from '@tanstack/react-query';
import { useSession } from 'next-auth/react';
import { useLocale } from 'next-intl';
import { useRouter } from 'next/navigation';
import { DataGrid, DataGridRow, DataGridRowValue } from '@/lib/idsk';
import { date, Message, Page, unwrap, MessageTag as MessageTagType, api, RequestError } from '@/utils';
import { Empty, Error, QueryHandler } from '@/components/core';
import MessageListSkeleton from './MessageListSkeleton';
import MessageListHeader from './MessageListHeader';
import MessageTag from './MessageTag';
import { useBreakpoint } from '@/hooks';
import MessageListPlaceholder from './MessageListPlaceholder';

interface MessageListProps {
  page: number;
  pageSize: number;
  search: string;
  onLoad: (data: Page<Message>) => void;
}

const MessageTags: React.FC<{ tags: MessageTagType[] }> = ({ tags }) => (
  <div className="flex gap-2.5 flex-wrap w-full tb2:w-fit tb2:justify-end">
    {tags.map((tag, index) => (
      <MessageTag size="small" key={`tag-${index + 0}`} color={tag.color}>
        {tag.value}
      </MessageTag>
    ))}
  </div>
);

const MessageList: React.FC<MessageListProps> = ({ page, pageSize, search, onLoad }) => {
  const locale = useLocale();
  const router = useRouter();

  const { data } = useSession();

  const query = useQuery<Page<Message>, RequestError>({
    queryKey: ['messages', page, pageSize, search],
    queryFn: () =>
      unwrap(
        axios.get(api.messages(data?.user.id || ''), {
          params: {
            page,
            'page-size': pageSize,
            sort: 'createdAt:',
            filter: `body=like='*${search}*',title=like='*${search}*'`
          }
        })
      )
  });

  useEffect(() => {
    if (query.status === 'success') {
      onLoad(query.data);
    }
  }, [query.status, query.data, onLoad]);

  const { match, Breakpoints } = useBreakpoint();
  const isDesktop = match(Breakpoints.dm1);

  return (
    <QueryHandler
      query={query}
      error={
        <MessageListPlaceholder>
          <Error transparent err={query.error} />
        </MessageListPlaceholder>
      }
      loading={<MessageListSkeleton pageSize={pageSize} />}
    >
      {(messagePage) =>
        messagePage.items.length > 0 ? (
          <DataGrid id="messages-datagrid" headRow={<MessageListHeader />}>
            {messagePage.items.map((message: Message) => (
              <DataGridRow
                key={message.id}
                buttonProps={{ onClick: () => router.push(`/spravy/${message.id}`) }}
                className="flex-col tb2:flex-row gap-0 tb2:gap-4"
              >
                <DataGridRowValue className="tb1:w-[16rem] flex-none">
                  <span className="font-bold tb2:font-normal truncate">{message.senderName}</span>
                </DataGridRowValue>
                {isDesktop ? (
                  <>
                    <DataGridRowValue className="max-w-lg dm1:max-w-sm w-full text-txt-grey justify-start">
                      <span className="truncate">{message.title}</span>
                    </DataGridRowValue>
                    <DataGridRowValue align="right" className="w-full overflow-hidden mr-8">
                      <MessageTags tags={message.tags || []} />
                    </DataGridRowValue>
                  </>
                ) : (
                  <DataGridRowValue className="flex flex-row flex-wrap w-full justify-between mb-2 tb2:mb-0">
                    <span className="text-txt-grey line-clamp-2">{message.title}</span>
                    <MessageTags tags={message.tags || []} />
                  </DataGridRowValue>
                )}
                <DataGridRowValue className="flex-none tb2:justify-end">
                  {date(message.createdAt, locale)}
                </DataGridRowValue>
              </DataGridRow>
            ))}
          </DataGrid>
        ) : (
          <MessageListPlaceholder>
            <div className="px-12 py-16 bg-white rounded-b-lg border-t border-neutral-300">
              <Empty title="Zatiaľ nemáte žiadne správy" />
            </div>
          </MessageListPlaceholder>
        )
      }
    </QueryHandler>
  );
};

export default MessageList;
