'use client';

import axios from 'axios';
import { useQueryClient, useQuery, useMutation } from '@tanstack/react-query';
import { useTranslations } from 'next-intl';
import { useState } from 'react';
import { Page, Subscription, RequestError, unwrap, api, Organization } from '@/utils';
import { useNotifications } from '@/hooks';

type MutationData = { id: string; isEmailEnabled: boolean };

const PAGE_SIZE = 50;

const useSubscriptionsEdit = (userId: string, organization: Organization | null) => {
  const [page, setPage] = useState(1);
  const t = useTranslations('profile_page');

  const notify = useNotifications();
  const queryClient = useQueryClient();

  const queryKey = ['subscriptions', page, organization, PAGE_SIZE];

  const query = useQuery<Page<Subscription>, RequestError>({
    queryKey,
    queryFn: () =>
      unwrap(
        axios.get(api.subscriptions(userId), {
          params: { filter: `organizationId==${organization?.id}`, page, 'page-size': PAGE_SIZE, sort: 'topicName;' }
        })
      )
  });

  const mutation = useMutation({
    mutationKey: ['subscription'],
    mutationFn: ({ id, isEmailEnabled }: MutationData) =>
      unwrap(axios.put(api.subscriptionDetail(userId, id), { isEmailEnabled })),
    onMutate: async ({ id, isEmailEnabled }: MutationData) => {
      await queryClient.cancelQueries({ queryKey });
      const prevSubscriptions = queryClient.getQueryData<Page<Subscription>>(queryKey);

      if (prevSubscriptions) {
        queryClient.setQueryData(queryKey, (old: Page<Subscription> | undefined) =>
          old
            ? {
                ...old,
                items: old.items.map((item) => (item.id === id ? { ...item, isEmailEnabled } : item))
              }
            : old
        );
      }

      return { prevSubscriptions };
    },
    onError: (_err, _mutationData, context: any) => {
      if (context?.prevSubscriptions) {
        queryClient.setQueryData(queryKey, context.prevSubscriptions);
      }

      notify({
        message: t('subscription_error'),
        variant: 'warning'
      });
    },
    onSettled: () => {
      queryClient.invalidateQueries({ queryKey });
    }
  });

  const handleToggle = (subscription: Subscription) => {
    mutation.mutate({ id: subscription.id, isEmailEnabled: !subscription.isEmailEnabled });
  };

  return { query, mutation, handleToggle, setPage, PAGE_SIZE, t };
};

export default useSubscriptionsEdit;
