'use client';

import React, { useMemo } from 'react';
import axios from 'axios';
import { useLocale, useTranslations } from 'next-intl';
import { useMutation } from '@tanstack/react-query';
import { QrCodeIcon } from '@slovakia-kosice/idsk-react';
import { TableRow, TableRowValue, TertiaryIconButton, DownloadIcon, TextButton } from '@/lib/idsk';
import { date, download, labels, Proceeding, replaceExtension, unwrap } from '@/utils';
import { useNotifications } from '@/hooks';
import { TableListAction, TableListItem, TableListValue } from '@/components/core';
import ProceedingsTags from './ProceedingsTags';

interface ProceedingItemProps {
  item: Proceeding;
  view: 'table' | 'list';
  type: 'tax' | 'payment';
  onPay: (item: Proceeding) => void;
}

const ProceedingItem: React.FC<ProceedingItemProps> = ({ item, view, type, onPay }) => {
  const notify = useNotifications();
  const t = useTranslations('taxes');

  const locale = useLocale();

  const mutation = useMutation({
    mutationKey: ['attachment'],
    mutationFn: (proceeding: any) =>
      unwrap(
        axios.get(`/api${proceeding.attachment.url}`, {
          responseType: 'arraybuffer',
          headers: {
            accept: 'application/octet-stream'
          }
        })
      ),
    onSuccess: (data, variables) => {
      const { attachment, title } = variables;
      const blob = new Blob([data], { type: 'application/pdf' });

      // Sometimes api returns pdf file with .asice extension
      download(blob, replaceExtension(attachment.title, 'pdf'));
      notify({
        message: t('download_success', { title })
      });
    },
    onError: () => {
      notify({
        message: t('download_failed'),
        variant: 'warning'
      });
    }
  });

  const buildStatuses = (proceeding: Proceeding) =>
    proceeding.statuses.map((status) => {
      const variant = Object.keys(labels).find((key) => labels[key].includes(status)) || 'basic';

      return {
        variant,
        status
      };
    });

  const buildAttachment = (metadata: any) => {
    const attachment = {
      id: metadata.id,
      title: '',
      mime: '',
      size: '',
      url: ''
    };

    metadata.forEach((element: any) => {
      if (element.key === 'NAZOV') {
        attachment.title = element.value;
      }

      if (element.key === 'MIME_TYPE') {
        attachment.mime = element.value;
      }

      if (element.key === 'VELKOST') {
        attachment.size = element.value;
      }

      if (element.key === 'DOWNLOAD_RELATIVE_URL') {
        attachment.url = element.value;
      }
    });

    return attachment;
  };

  const taxItem = useMemo(() => {
    const object: any = {
      ...item,
      statuses: buildStatuses(item)
    };

    item.sources.forEach((source) => {
      if (source.key === 'CENA_CELKOM') {
        object.total = source.value;
      }

      if (source.key === 'UHRADENE') {
        object.paid = source.value;
      }

      if (source.key === 'PRILOHA') {
        object.attachment = buildAttachment(source.metadata);
      }
    });

    return object;
  }, [item]);

  if (view === 'table') {
    return (
      <TableRow className="flex w-full">
        <TableRowValue className="w-[25%] tb2:w-[25%] dm1:w-[20%]">{taxItem.title}</TableRowValue>
        <TableRowValue className="w-[21%] tb2:w-[20%] dm1:w-[15%]">{date(taxItem.created, locale)}</TableRowValue>
        <TableRowValue className="w-[17%] tb2:w-[22%] dm1:w-[11%]">{taxItem.total} EUR</TableRowValue>
        <TableRowValue className="w-[17%] tb2:w-[21%] dm1:w-[11%]">{taxItem.paid} EUR</TableRowValue>
        <TableRowValue className="w-[13%] tb2:w-[12%] dm1:w-[43%] flex gap-2 last:!static" align="right">
          <ProceedingsTags item={taxItem} />
          <TertiaryIconButton icon={<QrCodeIcon />} onClick={() => onPay(taxItem)} />
          <TertiaryIconButton icon={<DownloadIcon />} onClick={() => mutation.mutate(taxItem)} />
        </TableRowValue>
      </TableRow>
    );
  }

  return (
    <TableListItem>
      <TableListValue label={t(`name_of_${type}`)}>{taxItem.title}</TableListValue>
      <TableListValue label={t('date_of_issue')}>{date(taxItem.created, locale)}</TableListValue>
      <TableListValue label={t('sum')}>{taxItem.total} EUR</TableListValue>
      <TableListValue label={t('paid')}>{taxItem.paid} EUR</TableListValue>
      <TableListAction>
        <TextButton icon={<QrCodeIcon />} className="text-black" onClick={() => onPay(taxItem)}>
          {t('pay')}
        </TextButton>
      </TableListAction>
      <TableListAction>
        <TextButton icon={<DownloadIcon />} iconPosition="right" onClick={() => mutation.mutate(taxItem)}>
          {t('download_document')}
        </TextButton>
      </TableListAction>
    </TableListItem>
  );
};

export default ProceedingItem;
