'use client';
import { usePathname } from 'next/navigation';
import { useState, useEffect } from 'react';

export type Cookies = {
  necessary: boolean;
  analytics: boolean;
  preferential: boolean;
};

type AtLeastOneCookie = Partial<Cookies> &
  {
    [K in keyof Cookies]: Required<Pick<Cookies, K>>;
  }[keyof Cookies];

const LOCALSTORAGE_KEY = 'kk-cookies';

const getCookiesFromLocalStorage = () => {
  if (typeof window !== 'undefined') {
    const stored = localStorage.getItem(LOCALSTORAGE_KEY);
    return stored ? (JSON.parse(stored) as Cookies) : null;
  }

  return null;
};

const useCookies = () => {
  const [mounted, setMounted] = useState(false);
  const [cookies, setCookies] = useState<Cookies | null>(getCookiesFromLocalStorage);

  const path = usePathname();

  const update = (settings: AtLeastOneCookie) => {
    const newSettings = { ...cookies, ...settings } as Cookies;
    localStorage.setItem(LOCALSTORAGE_KEY, JSON.stringify(newSettings));
    setCookies(newSettings);

    const gtagValue = settings?.analytics ? 'granted' : 'denied';
    window.gtag('consent', 'update', {
      analytics_storage: gtagValue
    });
  };

  useEffect(() => {
    setMounted(true);
    setCookies(getCookiesFromLocalStorage());
  }, [path]);

  const displayBanner = mounted && cookies === null && !path.includes('cookies');

  return { cookies, displayBanner, update };
};

export default useCookies;
