import { AxiosError, isAxiosError } from 'axios';
import { ZodError } from 'zod';
import { ApiError, Fault, RequestError, SimpleApiError } from '../../types';

const serverNotResponding = (err: AxiosError): RequestError => ({
  status: 500,
  statusText: 'Internal Server Error',
  message: err.message,
  correlationId: '',
  fault: [
    {
      serviceId: 'web_app_spravca_kosice',
      faultCode: 'SERVER_NOT_RESPONDING',
      faultMessage: 'The request was made but no response was received.',
      faultMessageFormat: []
    }
  ]
});

const requestSetupError = (err: AxiosError): RequestError => ({
  status: 500,
  statusText: 'Internal Server Error',
  message: err.message,
  correlationId: '',
  fault: [
    {
      serviceId: 'web_app_spravca_kosice',
      faultCode: 'REQUEST_SETUP_ERROR',
      faultMessage: 'An error occurred while setting up the http request.',
      faultMessageFormat: []
    }
  ]
});

const parseError = (err: ZodError): RequestError => {
  const faults = err.errors.map((item) => ({
    serviceId: 'web_app_spravca_kosice',
    faultCode: item.code.toUpperCase(),
    faultMessage: `Error: ${item.message} at: ${item.path}`,
    faultMessageFormat: []
  }));

  return {
    status: 500,
    statusText: 'Parsing Error',
    message: err.message,
    correlationId: '',
    fault: faults
  };
};

const unknownError = (err: unknown): RequestError => {
  let message = 'Unknown error has occured.';

  if (typeof err === 'string') {
    message = err;
  }

  if (!!err && typeof err === 'object' && 'message' in err && typeof err.message === 'string') {
    message = err.message;
  }

  return {
    status: 500,
    statusText: 'Internal Server Error',
    message,
    correlationId: '',
    fault: [
      {
        serviceId: 'web_app_spravca_kosice',
        faultCode: 'UNKNOWN_ERROR',
        faultMessage: 'Unknown error has occured. Please contact technical support.',
        faultMessageFormat: []
      }
    ]
  };
};

// Check if { correlationId: xxx-xxx, fault: []} error was returned from the server
const isApiError = (data: unknown): data is ApiError => {
  if (!!data && typeof data === 'object' && 'correlationId' in data && 'fault' in data) {
    return true;
  }
  return false;
};

// Check if { correlationId: xxx-xxx, serviceId: xxx, faultCode: xxx, faultMessage: xxx, faultMessageFormat: []} error was returned from the server
const isSimpleApiError = (data: unknown): data is SimpleApiError => {
  if (
    !!data &&
    typeof data === 'object' &&
    'correlationId' in data &&
    'serviceId' in data &&
    'faultCode' in data &&
    'faultMessage' in data &&
    'faultMessageParams' in data
  ) {
    return true;
  }
  return false;
};

export const isRequestError = (data: unknown): data is RequestError => {
  const keys = ['status', 'statusText', 'message', 'correlationId', 'fault'];
  if (!!data && typeof data === 'object') {
    return keys.every((key) => key in data);
  }
  return false;
};

export const getRequestError = (err: unknown): RequestError => {
  if (isAxiosError(err)) {
    if (err.response) {
      // The request was made and the server responded with a status code that falls out of range of 2xx
      const { data } = err.response;
      const correlationId = data.correlationId || '';

      let fault: Fault[] = [
        {
          serviceId: 'web_app_spravca_kosice',
          faultCode: 'UNKNOWN_HTTP_REQUEST_ERROR',
          faultMessage: 'Error body returned by the server can not be recognized. Please contact technical support.',
          faultMessageFormat: []
        }
      ];

      if (isApiError(data)) {
        fault = data.fault;
      } else if (isSimpleApiError(data)) {
        fault = [
          {
            serviceId: data.serviceId,
            faultCode: data.faultCode,
            faultMessage: data.faultMessage,
            faultMessageFormat: data.faultMessageParams
          }
        ];
      }

      const error: RequestError = {
        status: err.response.status,
        statusText: err.response.statusText,
        message: err.message,
        correlationId,
        fault
      };

      return error;
    }
    if (err.request) {
      return serverNotResponding(err); // The request was made but no response was received
    }
    return requestSetupError(err); // Something happened in setting up the request that triggered an Error
  }

  if (err instanceof ZodError) {
    return parseError(err); // Zod parsing failed
  }
  return unknownError(err); // Is not axios error
};
