package sk.kosice.konto.kknotificationservice.restapi.config;

import static sk.kosice.konto.kknotificationservice.restapi.controller.RecipientController.RECIPIENT_URI;

import com.fasterxml.jackson.databind.ObjectMapper;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Optional;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configurers.AbstractHttpConfigurer;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.jwt.JwtDecoders;
import org.springframework.security.oauth2.jwt.NimbusJwtDecoder;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
import sk.kosice.konto.kknotificationservice.restapi.controller.BaseController;
import sk.kosice.konto.kknotificationservice.restapi.dto.common.error.ErrorDetailResponse;
import sk.kosice.konto.kknotificationservice.restapi.error.ApiErrorCode;

@Configuration
@EnableWebSecurity
public class SecurityConfiguration {

  public static final String AUTHENTICATION_SCHEME_BEARER = "Bearer";

  @Bean(name = "ocAppIntegrationJwtDecoder")
  public JwtDecoder ocAppIntegrationJwtDecoder(OAuthPropertiesPort oAuthPropertiesPort) {
    if (oAuthPropertiesPort.ocAppIntegrationIssuerLocation() != null
        && !oAuthPropertiesPort.ocAppIntegrationIssuerLocation().isBlank()
        && !oAuthPropertiesPort.ocAppIntegrationIssuerLocation().isEmpty()) {
      return JwtDecoders.fromIssuerLocation(oAuthPropertiesPort.ocAppIntegrationIssuerLocation());
    } else {
      return NimbusJwtDecoder.withJwkSetUri(oAuthPropertiesPort.ocAppIntegrationJwksUrl()).build();
    }
  }

  RequestMatcher[] appIntegrationOAuthEndpoints = {
    new AntPathRequestMatcher(RECIPIENT_URI, HttpMethod.GET.name()),
    new AntPathRequestMatcher(RECIPIENT_URI, HttpMethod.PUT.name())
  };

  @Bean
  public SecurityFilterChain filterChainOAuthAppIntegration(
      HttpSecurity http, @Qualifier("ocAppIntegrationJwtDecoder") JwtDecoder jwtDecoder)
      throws Exception {

    http.csrf(AbstractHttpConfigurer::disable)
        .cors(AbstractHttpConfigurer::disable)
        .securityMatchers((matchers) -> matchers.requestMatchers(appIntegrationOAuthEndpoints))
        .authorizeHttpRequests(
            auth -> auth.requestMatchers(appIntegrationOAuthEndpoints).authenticated())
        .oauth2ResourceServer(oauth2 -> oauth2.jwt(jwt -> jwt.decoder(jwtDecoder)));

    // Invokes when a server receives valid credentials that are not adequate to gain access.
    http.oauth2ResourceServer(
        configurer ->
            configurer.authenticationEntryPoint(
                invalidOrInsufficientCredentialsEntryPoint(AUTHENTICATION_SCHEME_BEARER)));

    // Invokes when the request lacks valid authentication credentials for the target resource.
    http.exceptionHandling(
        configurer ->
            configurer.authenticationEntryPoint(
                missingCredentialsEntryPoint(AUTHENTICATION_SCHEME_BEARER)));

    return http.build();
  }

  private AuthenticationEntryPoint missingCredentialsEntryPoint(String authenticationScheme) {
    return missingCredentialsEntryPoint(Optional.ofNullable(authenticationScheme));
  }

  private AuthenticationEntryPoint missingCredentialsEntryPoint(
      Optional<String> authenticationScheme) {
    return (request, response, authException) ->
        setUnauthorizedApiErrorToHttpServletResponse(response, authenticationScheme);
  }

  private AuthenticationEntryPoint invalidOrInsufficientCredentialsEntryPoint(
      String authenticationScheme) {
    return invalidOrInsufficientCredentialsEntryPoint(Optional.ofNullable(authenticationScheme));
  }

  private AuthenticationEntryPoint invalidOrInsufficientCredentialsEntryPoint(
      Optional<String> authenticationScheme) {
    return (request, response, authException) -> {
      if (authException instanceof AuthenticationException) {
        setUnauthorizedApiErrorToHttpServletResponse(response, authenticationScheme);
      } else {
        setApiErrorToHttpServletResponse(response, ApiErrorCode.FORBIDDEN);
      }
    };
  }

  private HttpServletResponse setUnauthorizedApiErrorToHttpServletResponse(
      HttpServletResponse response, Optional<String> authenticationScheme) throws IOException {
    final HttpServletResponse servletResponse =
        setApiErrorToHttpServletResponse(response, ApiErrorCode.UNAUTHORIZED);
    authenticationScheme.ifPresent(s -> servletResponse.setHeader(HttpHeaders.WWW_AUTHENTICATE, s));
    return servletResponse;
  }

  private HttpServletResponse setApiErrorToHttpServletResponse(
      HttpServletResponse response, ApiErrorCode errorCode) throws IOException {
    final ResponseEntity<ErrorDetailResponse> responseEntity =
        BaseController.mapErrorToResponse(errorCode.createError());
    response.setStatus(responseEntity.getStatusCode().value());
    response.setContentType(MediaType.APPLICATION_JSON_VALUE);
    String jsonBody = new ObjectMapper().writeValueAsString(responseEntity.getBody());
    response.getOutputStream().write(jsonBody.getBytes(StandardCharsets.UTF_8));
    return response;
  }
}
