package sk.kosice.konto.kknotificationservice.restapi.controller;

import static sk.kosice.konto.kknotificationservice.restapi.config.ModuleWebApiConfig.BEARER_AUTH;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.extensions.Extension;
import io.swagger.v3.oas.annotations.extensions.ExtensionProperty;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import java.util.Optional;
import java.util.UUID;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RestController;
import sk.kosice.konto.kknotificationservice.business.recipient.port.inbound.CreateOrUpdateRecipientUseCase;
import sk.kosice.konto.kknotificationservice.business.recipient.port.inbound.FindRecipientByIdUseCase;
import sk.kosice.konto.kknotificationservice.domain.common.ImmutableActorIdentity;
import sk.kosice.konto.kknotificationservice.domain.recipient.query.FindRecipientByIdQuery;
import sk.kosice.konto.kknotificationservice.restapi.command.RecipientCommandFactory;
import sk.kosice.konto.kknotificationservice.restapi.common.enumeration.Platform;
import sk.kosice.konto.kknotificationservice.restapi.dto.recipient.CreateOrUpdateRecipientRequest;
import sk.kosice.konto.kknotificationservice.restapi.dto.recipient.RecipientDetailResponse;
import sk.kosice.konto.kknotificationservice.restapi.mapper.RecipientResponseMapper;

@Tag(name = "Recipient", description = "Rest API for recipient operations.")
@RestController
@Validated
public class RecipientController extends BaseController {

  public static final String RECIPIENT_URI = BASE_V1_URI + "/recipients/{kid}";

  private final FindRecipientByIdUseCase findRecipientByIdUseCase;
  private final CreateOrUpdateRecipientUseCase createOrUpdateRecipientUseCase;

  @Autowired
  public RecipientController(
      FindRecipientByIdUseCase findRecipientByIdUseCase,
      CreateOrUpdateRecipientUseCase createOrUpdateRecipientUseCase) {
    this.findRecipientByIdUseCase = findRecipientByIdUseCase;
    this.createOrUpdateRecipientUseCase = createOrUpdateRecipientUseCase;
  }

  @Operation(
      summary = "Detail preferred contact values for Recipient (KKProfile).",
      security = @SecurityRequirement(name = BEARER_AUTH),
      extensions = {
        @Extension(
            properties = {
              @ExtensionProperty(
                  name = "x-eventcatalog-message-name",
                  value = "Detail preferred Contact")
            })
      })
  @ApiResponses({
    @ApiResponse(
        responseCode = "200",
        description =
            "Successful operation - response with stored details of recipient in kknotification service.",
        content = @Content(schema = @Schema(implementation = RecipientDetailResponse.class)))
  })
  @GetMapping(value = RECIPIENT_URI)
  public ResponseEntity<?> getDetail(
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of recipient kid.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_KID)
          UUID kid,
      @AuthenticationPrincipal Jwt jwt) {
    return ResponseEntity.ok(
        RecipientResponseMapper.map(
            findRecipientByIdUseCase.execute(
                FindRecipientByIdQuery.of(
                    kid,
                    Optional.ofNullable(getStringClaim(jwt, EMPLOYEE_ID_CLAIM_NAME))
                        .map(ImmutableActorIdentity::of)))));
  }

  @Operation(
      summary = "Create or update preferred mail for Recipient (KKProfile).",
      security = @SecurityRequirement(name = BEARER_AUTH),
      requestBody =
          @io.swagger.v3.oas.annotations.parameters.RequestBody(
              description = "Body of request for storing preferred mail for KKProfile.",
              required = true,
              content =
                  @Content(
                      schema = @Schema(implementation = CreateOrUpdateRecipientRequest.class))),
      extensions = {
        @Extension(
            properties = {
              @ExtensionProperty(
                  name = "x-eventcatalog-message-name",
                  value = "Set preferred Contact")
            })
      })
  @ApiResponses({
    @ApiResponse(
        responseCode = "200",
        description =
            "Successful operation - response with stored details of recipient in kknotification service.",
        content = @Content(schema = @Schema(implementation = RecipientDetailResponse.class)))
  })
  @PutMapping(value = RECIPIENT_URI)
  public ResponseEntity<?> createOrUpdate(
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of recipient kid.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_KID)
          UUID kid,
      @Valid @RequestBody CreateOrUpdateRecipientRequest request,
      @AuthenticationPrincipal Jwt jwt) {
    return ResponseEntity.ok(
        RecipientResponseMapper.map(
            createOrUpdateRecipientUseCase.execute(
                RecipientCommandFactory.createOrUpdateRecipientCommand(
                    kid,
                    request,
                    Optional.ofNullable(getStringClaim(jwt, EMPLOYEE_ID_CLAIM_NAME))))));
  }
}
