package sk.kosice.konto.kkmessageservice.restapi.controller;

import static sk.kosice.konto.kkmessageservice.business.AbstractService.SERVICE_NAME;
import static sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingRequestParams.PARAM_FILTER;
import static sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingRequestParams.PARAM_PAGE;
import static sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingRequestParams.PARAM_PAGE_SIZE;
import static sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingRequestParams.PARAM_SORT;
import static sk.kosice.konto.kkmessageservice.restapi.command.MessageCommandFactory.createMessageCommand;

import com.neovisionaries.i18n.LanguageCode;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.extensions.Extension;
import io.swagger.v3.oas.annotations.extensions.ExtensionProperty;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.ExampleObject;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import sk.kosice.konto.kkmessageservice.business.message.port.inbound.CreateMessageUseCase;
import sk.kosice.konto.kkmessageservice.business.message.port.inbound.FindMessageByIdAsOrganizationUseCase;
import sk.kosice.konto.kkmessageservice.business.message.port.inbound.FindMessageByIdAsRecipientUseCase;
import sk.kosice.konto.kkmessageservice.business.message.port.inbound.ListMessageUseCase;
import sk.kosice.konto.kkmessageservice.domain.common.audit.ImmutableActorIdentity;
import sk.kosice.konto.kkmessageservice.domain.common.listing.MessageListingAttribute;
import sk.kosice.konto.kkmessageservice.domain.message.query.FindMessageByIdAndKidQuery;
import sk.kosice.konto.kkmessageservice.domain.message.query.FindMessageByIdAndOrganizationIdQuery;
import sk.kosice.konto.kkmessageservice.domain.message.query.ImmutableFindMessageByIdAndKidQuery;
import sk.kosice.konto.kkmessageservice.domain.message.query.ImmutableFindMessageByIdAndOrganizationIdQuery;
import sk.kosice.konto.kkmessageservice.domain.message.query.MessageListingQuery;
import sk.kosice.konto.kkmessageservice.restapi.command.MessageQueryFactory;
import sk.kosice.konto.kkmessageservice.restapi.common.enumeration.Platform;
import sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode;
import sk.kosice.konto.kkmessageservice.restapi.common.listing.ListFilteringParser;
import sk.kosice.konto.kkmessageservice.restapi.common.listing.ListOrderingParser;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ImmutableErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ImmutableErrorDetailResponseItem;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageCreateRequest;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageListResponse;
import sk.kosice.konto.kkmessageservice.restapi.mapper.MessageResponseMapper;
import sk.kosice.konto.kkmessageservice.restapi.springdoc.restapi.springdoc.annotation.FilterParameter;
import sk.kosice.konto.kkmessageservice.restapi.springdoc.restapi.springdoc.annotation.SortParameter;

@Tag(name = "Message", description = "Rest API for message operations.")
@RestController
@Validated
public class MessageController extends BaseController {

  public static final String MESSAGES_URI = BASE_V1_URI + "/recipients/{kid}/messages";
  public static final String MESSAGE_URI = MESSAGES_URI + "/{messageId}";

  public static final String MESSAGES_BY_ORGANIZATION_URI =
      BASE_V1_URI + "/organizations/{organizationId}/messages";
  public static final String MESSAGE_BY_ORGANIZATION_URI =
      MESSAGES_BY_ORGANIZATION_URI + "/{messageId}";

  private final FindMessageByIdAsRecipientUseCase findMessageByIdAsRecipientUseCase;
  private final FindMessageByIdAsOrganizationUseCase findMessageByIdAsOrganizationUseCase;
  private final ListMessageUseCase listMessageUseCase;
  private final CreateMessageUseCase createMessageUseCase;

  private final ListFilteringParser listFilteringParser;
  private final ListOrderingParser listOrderingParser;

  @Autowired
  public MessageController(
      FindMessageByIdAsRecipientUseCase findMessageByIdAsRecipientUseCase,
      FindMessageByIdAsOrganizationUseCase findMessageByIdAsOrganizationUseCase,
      ListMessageUseCase listMessageUseCase,
      CreateMessageUseCase createMessageUseCase,
      ListFilteringParser listFilteringParser,
      ListOrderingParser listOrderingParser) {
    this.findMessageByIdAsRecipientUseCase = findMessageByIdAsRecipientUseCase;
    this.findMessageByIdAsOrganizationUseCase = findMessageByIdAsOrganizationUseCase;
    this.listMessageUseCase = listMessageUseCase;
    this.createMessageUseCase = createMessageUseCase;
    this.listFilteringParser = listFilteringParser;
    this.listOrderingParser = listOrderingParser;
  }

  @Operation(
      summary = "Message detail.",
      description = "Returns detail of the message given by its ID.",
      extensions = {
        @Extension(
            properties = {
              @ExtensionProperty(
                  name = "x-eventcatalog-message-name",
                  value = "Message by Recipient")
            })
      })
  @ApiResponses({
    @ApiResponse(
        responseCode = "200",
        description = "Successful operation - response with details of the message.",
        content = @Content(schema = @Schema(implementation = MessageDetailResponse.class))),
    @ApiResponse(
        responseCode = "404",
        description = "Message was not found.",
        content =
            @Content(
                schema = @Schema(implementation = ErrorDetailResponse.class),
                examples = {
                  @ExampleObject(
                      name = "MESSAGE_DOES_NOT_EXIST",
                      description = "Message with id does not exist.",
                      value =
                          """
                {
                   "correlationId":"da353dcd-c6c7-4ecd-8660-0cbd8e558ad9",
                   "serviceId": "kkmessage-service",
                   "faultCode": "MESSAGE_DOES_NOT_EXIST",
                   "faultMessage": "Message with id '%s' does not exist.",
                   "faultMessageParams": [
                      "FindMessageByIdQuery{id=e3f965f9-2975-4c5e-a94a-628d49acff76, recipientKid=84b3ac2f-18ff-40fa-b8c9-ed29d774a7ff}"
                   ]

                }
                """)
                }))
  })
  @GetMapping(value = MESSAGE_URI, produces = MediaType.APPLICATION_JSON_VALUE)
  public ResponseEntity<?> detailByRecipient(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of recipient kid.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(value = PARAM_KID)
          UUID kid,
      @Parameter(
              description = "Unique identifier of message.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_MESSAGE_ID)
          UUID messageId) {
    log.info("Calling MessageController GET detail: [{}]", MESSAGE_URI);

    final var kidFromAuth = getStringClaim(jwt, EXTENSION_KID_CLAIM_NAME);
    if (!kidFromAuth.equals(kid.toString())) {
      return ResponseEntity.status(HttpStatus.FORBIDDEN)
          .body(
              ImmutableErrorDetailResponse.builder()
                  .correlationId(UUID.randomUUID().toString())
                  .addFault(
                      ImmutableErrorDetailResponseItem.builder()
                          .serviceId(SERVICE_NAME)
                          .faultCode(BaseApiErrorCode.BAD_REQUEST.toString())
                          .faultMessage("You are not authorized to access this resource.")
                          .build())
                  .build());
    }

    final FindMessageByIdAndKidQuery query =
        ImmutableFindMessageByIdAndKidQuery.builder().id(messageId).recipientKid(kid).build();
    return ResponseEntity.ok(
        MessageResponseMapper.map(findMessageByIdAsRecipientUseCase.execute(query)));
  }

  @Operation(
      summary = "List of messages.",
      description = "Returns a list of messages matching given conditions.",
      extensions = {
        @Extension(
            properties = {
              @ExtensionProperty(
                  name = "x-eventcatalog-message-name",
                  value = "Messages by Recipient")
            })
      })
  @ApiResponses({
    @ApiResponse(
        responseCode = "200",
        description = "Successful response with messages matching given conditions.",
        content = @Content(schema = @Schema(implementation = MessageListResponse.class)))
  })
  @GetMapping(value = MESSAGES_URI, produces = MediaType.APPLICATION_JSON_VALUE)
  public ResponseEntity<?> list(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of recipient kid.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_KID)
          UUID kid,
      @FilterParameter(
              listingAttributeClass = MessageListingAttribute.class,
              example = "body=like=*Pozvánka*")
          @RequestParam(name = PARAM_FILTER, defaultValue = "")
          String filter,
      @Parameter(description = "Page query parameter.")
          @RequestParam(name = PARAM_PAGE, defaultValue = "1")
          Integer page,
      @Parameter(description = "Page size query parameter.")
          @RequestParam(name = PARAM_PAGE_SIZE, defaultValue = "10")
          Integer pageSize,
      @SortParameter(listingAttributeClass = MessageListingAttribute.class)
          @RequestParam(name = PARAM_SORT, defaultValue = "")
          String orderings) {
    log.info("Calling MessageController GET list: [{}]", MESSAGES_URI);

    final var kidFromAuth = getStringClaim(jwt, EXTENSION_KID_CLAIM_NAME);
    if (!kidFromAuth.equals(kid.toString())) {
      return ResponseEntity.status(HttpStatus.FORBIDDEN)
          .body(
              ImmutableErrorDetailResponse.builder()
                  .correlationId(UUID.randomUUID().toString())
                  .addFault(
                      ImmutableErrorDetailResponseItem.builder()
                          .serviceId(SERVICE_NAME)
                          .faultCode(BaseApiErrorCode.BAD_REQUEST.toString())
                          .faultMessage("You are not authorized to access this resource.")
                          .build())
                  .build());
    }

    final MessageListingQuery query =
        MessageQueryFactory.map(
            kid,
            null,
            Optional.empty(),
            listFilteringParser.parse(filter, List.of(MessageListingAttribute.values())),
            listOrderingParser.parse(
                orderings, List.of(MessageListingAttribute.values()), new LanguageCode[] {}),
            page,
            pageSize,
            100);

    return ResponseEntity.ok(MessageResponseMapper.map(listMessageUseCase.execute(query)));
  }

  @Operation(
      summary = "Creates a personalized message.",
      description = "Creates a personalized message.",
      requestBody =
          @io.swagger.v3.oas.annotations.parameters.RequestBody(
              description = "Body of request for creating personalized message.",
              required = true,
              content = @Content(schema = @Schema(implementation = MessageCreateRequest.class))),
      extensions = {
        @Extension(
            properties = {
              @ExtensionProperty(
                  name = "x-eventcatalog-message-name",
                  value = "Message create personalized")
            })
      })
  @ApiResponses({
    @ApiResponse(
        responseCode = "201",
        description =
            "Successful operation - response with details of the created personalized message.",
        content = @Content(schema = @Schema(implementation = MessageDetailResponse.class))),
    @ApiResponse(
        responseCode = "404",
        description = "Recipient was not found",
        content =
            @Content(
                schema = @Schema(implementation = ErrorDetailResponse.class),
                examples = {
                  @ExampleObject(
                      name = "RECIPIENT_DOES_NOT_EXIST",
                      description = "Recipient with kid does not exist.",
                      value =
                          """
                {
                  "correlationId":"f99d9a88-589e-40aa-90fc-ddf75a9bb478",
                  "serviceId": "kkmessage-service",
                  "faultCode": "RECIPIENT_DOES_NOT_EXIST",
                  "faultMessage": "Recipient with kid '%s' does not exist.",
                  "faultMessageParams": [
                     "fd6c405f-7ae5-4d97-816e-a929c04b7001"
                  ]
                }
                """)
                }))
  })
  @PostMapping(value = MESSAGES_URI)
  public ResponseEntity<?> createMessage(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.onBehalfOf.description",
              required = true,
              example = "b911a44b-0b29-4a87-ba86-d6acc18fa38f")
          @RequestHeader(name = HEADER_X_ON_BEHALF_OF)
          UUID onBehalfOf,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of recipient kid.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_KID)
          UUID kid,
      @Valid @RequestBody MessageCreateRequest request) {
    log.info("Calling MessageController POST create: [{}]; request ({})", MESSAGES_URI, request);

    // TODO: sender name will be parsed from jwt token
    return new ResponseEntity<>(
        MessageResponseMapper.map(
            createMessageUseCase.execute(
                createMessageCommand(
                        onBehalfOf,
                        kid,
                        Optional.of(getStringClaim(jwt, EMPLOYEE_ID_CLAIM_NAME)),
                        request)
                    .withRecipientKid(kid))),
        HttpStatus.CREATED);
  }

  @Operation(
      summary = "Creates a public message.",
      description = "Creates a public message.",
      requestBody =
          @io.swagger.v3.oas.annotations.parameters.RequestBody(
              description = "Body of request for creating public message.",
              required = true,
              content = @Content(schema = @Schema(implementation = MessageCreateRequest.class))),
      extensions = {
        @Extension(
            properties = {
              @ExtensionProperty(
                  name = "x-eventcatalog-message-name",
                  value = "Message create public")
            })
      })
  @ApiResponses({
    @ApiResponse(
        responseCode = "201",
        description = "Successful operation - response with details of the created public message.",
        content = @Content(schema = @Schema(implementation = MessageDetailResponse.class)))
  })
  @PostMapping(value = MESSAGES_BY_ORGANIZATION_URI)
  public ResponseEntity<?> createPublic(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.onBehalfOf.description",
              required = true,
              example = "b911a44b-0b29-4a87-ba86-d6acc18fa38f")
          @RequestHeader(name = HEADER_X_ON_BEHALF_OF)
          UUID onBehalfOf,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of organization id.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_ORGANIZATION_ID)
          UUID organizationId,
      @Valid @RequestBody MessageCreateRequest request) {
    log.info(
        "Calling MessageController POST create public: [{}]; request ({})",
        MESSAGES_BY_ORGANIZATION_URI,
        request);

    // TODO: sender name will be parsed from jwt token
    return new ResponseEntity<>(
        MessageResponseMapper.map(
            createMessageUseCase.execute(
                createMessageCommand(
                        organizationId,
                        null,
                        Optional.of(getStringClaim(jwt, EMPLOYEE_ID_CLAIM_NAME)),
                        request)
                    .withOrganizationId(organizationId))),
        HttpStatus.CREATED);
  }

  @Operation(
      summary = "List of messages by organization.",
      description = "Returns a list of messages by organization matching given conditions.")
  @ApiResponses({
    @ApiResponse(
        responseCode = "200",
        description = "Successful response with messages matching given conditions.",
        content = @Content(schema = @Schema(implementation = MessageListResponse.class)))
  })
  @GetMapping(value = MESSAGES_BY_ORGANIZATION_URI, produces = MediaType.APPLICATION_JSON_VALUE)
  public ResponseEntity<?> listByOrganization(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of recipient organization id.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_ORGANIZATION_ID)
          UUID organizationId,
      @FilterParameter(
              listingAttributeClass = MessageListingAttribute.class,
              example = "body=like=*Pozvánka*")
          @RequestParam(name = PARAM_FILTER, defaultValue = "")
          String filter,
      @Parameter(description = "Page query parameter.")
          @RequestParam(name = PARAM_PAGE, defaultValue = "1")
          Integer page,
      @Parameter(description = "Page size query parameter.")
          @RequestParam(name = PARAM_PAGE_SIZE, defaultValue = "10")
          Integer pageSize,
      @SortParameter(listingAttributeClass = MessageListingAttribute.class)
          @RequestParam(name = PARAM_SORT, defaultValue = "")
          String orderings) {
    log.info(
        "Calling MessageController GET list by organization: [{}]", MESSAGES_BY_ORGANIZATION_URI);

    final MessageListingQuery query =
        MessageQueryFactory.map(
            null,
            organizationId,
            Optional.of(getStringClaim(jwt, EMPLOYEE_ID_CLAIM_NAME)),
            listFilteringParser.parse(filter, List.of(MessageListingAttribute.values())),
            listOrderingParser.parse(
                orderings, List.of(MessageListingAttribute.values()), new LanguageCode[] {}),
            page,
            pageSize,
            20);

    return ResponseEntity.ok(MessageResponseMapper.mapWithKid(listMessageUseCase.execute(query)));
  }

  @GetMapping(value = MESSAGE_BY_ORGANIZATION_URI, produces = MediaType.APPLICATION_JSON_VALUE)
  public ResponseEntity<?> detailByOrganization(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of recipient organization id.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_ORGANIZATION_ID)
          UUID organizationId,
      @Parameter(
              description = "Unique identifier of message.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_MESSAGE_ID)
          UUID messageId) {
    log.info(
        "Calling MessageController GET detail by organization: [{}]", MESSAGE_BY_ORGANIZATION_URI);

    final FindMessageByIdAndOrganizationIdQuery query =
        ImmutableFindMessageByIdAndOrganizationIdQuery.builder()
            .id(messageId)
            .organizationId(organizationId)
            .actorId(ImmutableActorIdentity.of(getStringClaim(jwt, EMPLOYEE_ID_CLAIM_NAME)))
            .build();

    return ResponseEntity.ok(
        MessageResponseMapper.mapWithKid(findMessageByIdAsOrganizationUseCase.execute(query)));
  }
}
