package sk.kosice.konto.kkmessageservice.restapi.controller;

import static sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingRequestParams.PARAM_FILTER;
import static sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingRequestParams.PARAM_PAGE;
import static sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingRequestParams.PARAM_PAGE_SIZE;
import static sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingRequestParams.PARAM_SORT;
import static sk.kosice.konto.kkmessageservice.restapi.command.TopicCommandFactory.createTopicCommand;
import static sk.kosice.konto.kkmessageservice.restapi.command.TopicCommandFactory.deleteTopicCommand;
import static sk.kosice.konto.kkmessageservice.restapi.command.TopicCommandFactory.updateTopicCommand;

import com.neovisionaries.i18n.LanguageCode;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.extensions.Extension;
import io.swagger.v3.oas.annotations.extensions.ExtensionProperty;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.ExampleObject;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import java.util.List;
import java.util.UUID;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import sk.kosice.konto.kkmessageservice.business.topic.port.inbound.CreateTopicUseCase;
import sk.kosice.konto.kkmessageservice.business.topic.port.inbound.DeleteTopicUseCase;
import sk.kosice.konto.kkmessageservice.business.topic.port.inbound.FindTopicByIdUseCase;
import sk.kosice.konto.kkmessageservice.business.topic.port.inbound.ListTopicUseCase;
import sk.kosice.konto.kkmessageservice.business.topic.port.inbound.UpdateTopicUseCase;
import sk.kosice.konto.kkmessageservice.domain.common.listing.TopicListingAttribute;
import sk.kosice.konto.kkmessageservice.domain.topic.query.FindTopicByIdQuery;
import sk.kosice.konto.kkmessageservice.domain.topic.query.TopicListingQuery;
import sk.kosice.konto.kkmessageservice.restapi.command.TopicQueryFactory;
import sk.kosice.konto.kkmessageservice.restapi.common.enumeration.Platform;
import sk.kosice.konto.kkmessageservice.restapi.common.listing.ListFilteringParser;
import sk.kosice.konto.kkmessageservice.restapi.common.listing.ListOrderingParser;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.topic.TopicCreateRequest;
import sk.kosice.konto.kkmessageservice.restapi.dto.topic.TopicDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.topic.TopicListResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.topic.TopicUpdateRequest;
import sk.kosice.konto.kkmessageservice.restapi.mapper.TopicResponseMapper;
import sk.kosice.konto.kkmessageservice.restapi.springdoc.restapi.springdoc.annotation.FilterParameter;
import sk.kosice.konto.kkmessageservice.restapi.springdoc.restapi.springdoc.annotation.SortParameter;

@Tag(name = "Topic", description = "Rest API for topic operations.")
@RestController
public class TopicController extends BaseController {
  public static final String TOPICS_URI = BASE_V1_URI + "/organizations/{organizationId}/topics";
  public static final String TOPIC_URI = TOPICS_URI + "/{topicId}";

  private final ListFilteringParser listFilteringParser;
  private final ListOrderingParser listOrderingParser;

  private final CreateTopicUseCase createTopicUseCase;
  private final UpdateTopicUseCase updateTopicUseCase;
  private final FindTopicByIdUseCase findTopicByIdUseCase;
  private final ListTopicUseCase listTopicUseCase;
  private final DeleteTopicUseCase deleteTopicUseCase;

  @Autowired
  public TopicController(
      ListFilteringParser listFilteringParser,
      ListOrderingParser listOrderingParser,
      CreateTopicUseCase createTopicUseCase,
      UpdateTopicUseCase updateTopicUseCase,
      FindTopicByIdUseCase findTopicByIdUseCase,
      ListTopicUseCase listTopicUseCase,
      DeleteTopicUseCase deleteTopicUseCase) {
    this.listFilteringParser = listFilteringParser;
    this.listOrderingParser = listOrderingParser;
    this.createTopicUseCase = createTopicUseCase;
    this.updateTopicUseCase = updateTopicUseCase;
    this.findTopicByIdUseCase = findTopicByIdUseCase;
    this.listTopicUseCase = listTopicUseCase;
    this.deleteTopicUseCase = deleteTopicUseCase;
  }

  @Operation(
      summary = "Topic detail.",
      description = "Returns detail of the topic given by its ID.",
      extensions = {
        @Extension(
            properties = {
              @ExtensionProperty(
                  name = "x-eventcatalog-message-name",
                  value = "Topic in Organization")
            })
      })
  @ApiResponses({
    @ApiResponse(
        responseCode = "200",
        description = "Successful operation - response with details of the topic.",
        content = @Content(schema = @Schema(implementation = TopicDetailResponse.class))),
    @ApiResponse(
        responseCode = "404",
        description = "Topic was not found.",
        content =
            @Content(
                schema = @Schema(implementation = ErrorDetailResponse.class),
                examples = {
                  @ExampleObject(
                      name = "TOPIC_DOES_NOT_EXIST",
                      description = "Topic with id does not exist.",
                      value =
                          """
    {
       "correlationId":"da353dcd-c6c7-4ecd-8660-0cbd8e558ad9",
       "serviceId": "kkmessage-service",
       "faultCode": "TOPIC_DOES_NOT_EXIST",
       "faultTopic": "Topic with id '%s' does not exist.",
       "faultTopicParams": [
          "FindTopicByIdQuery{id=e3f965f9-2975-4c5e-a94a-628d49acff76, recipientKid=84b3ac2f-18ff-40fa-b8c9-ed29d774a7ff}"
       ]

    }
    """)
                }))
  })
  @GetMapping(value = TOPIC_URI, produces = MediaType.APPLICATION_JSON_VALUE)
  public ResponseEntity<?> detail(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of organization.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(value = PARAM_ORGANIZATION_ID)
          UUID organizationId,
      @Parameter(
              description = "Unique identifier of topic.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_TOPIC_ID)
          UUID topicId) {
    log.info("Calling TopicController GET detail: [{}]", TOPIC_URI);

    final FindTopicByIdQuery query =
        FindTopicByIdQuery.of(topicId, organizationId, getStringClaim(jwt, EMPLOYEE_ID_CLAIM_NAME));
    return ResponseEntity.ok(TopicResponseMapper.map(findTopicByIdUseCase.execute(query)));
  }

  @Operation(
      summary = "List of topics.",
      description = "Returns a list of topics matching given conditions.",
      extensions = {
        @Extension(
            properties = {
              @ExtensionProperty(
                  name = "x-eventcatalog-message-name",
                  value = "Topics in Organization")
            })
      })
  @ApiResponses({
    @ApiResponse(
        responseCode = "200",
        description = "Successful response with topics matching given conditions.",
        content = @Content(schema = @Schema(implementation = TopicListResponse.class)))
  })
  @GetMapping(value = TOPICS_URI, produces = MediaType.APPLICATION_JSON_VALUE)
  public ResponseEntity<?> list(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of organization.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_ORGANIZATION_ID)
          UUID organizationId,
      @FilterParameter(
              listingAttributeClass = TopicListingAttribute.class,
              example = "body=like=*Pozvánka*")
          @RequestParam(name = PARAM_FILTER, defaultValue = "")
          String filter,
      @Parameter(description = "Page query parameter.")
          @RequestParam(name = PARAM_PAGE, defaultValue = "1")
          Integer page,
      @Parameter(description = "Page size query parameter.")
          @RequestParam(name = PARAM_PAGE_SIZE, defaultValue = "10")
          Integer pageSize,
      @SortParameter(listingAttributeClass = TopicListingAttribute.class)
          @RequestParam(name = PARAM_SORT, defaultValue = "")
          String orderings) {
    log.info("Calling TopicController GET list: [{}]", TOPICS_URI);

    final TopicListingQuery query =
        TopicQueryFactory.map(
            organizationId,
            getStringClaim(jwt, EMPLOYEE_ID_CLAIM_NAME),
            listFilteringParser.parse(filter, List.of(TopicListingAttribute.values())),
            listOrderingParser.parse(
                orderings, List.of(TopicListingAttribute.values()), new LanguageCode[] {}),
            page,
            pageSize,
            100);

    return ResponseEntity.ok(TopicResponseMapper.map(listTopicUseCase.execute(query)));
  }

  @Operation(
      summary = "Creates a topic.",
      description = "Creates a topic.",
      requestBody =
          @io.swagger.v3.oas.annotations.parameters.RequestBody(
              description = "Body of request for creating topic.",
              required = true,
              content = @Content(schema = @Schema(implementation = TopicCreateRequest.class))),
      extensions = {
        @Extension(
            properties = {
              @ExtensionProperty(name = "x-eventcatalog-message-name", value = "Topic - Create")
            })
      })
  @ApiResponses({
    @ApiResponse(
        responseCode = "201",
        description = "Successful operation - response with details of the created topic.",
        content = @Content(schema = @Schema(implementation = TopicDetailResponse.class))),
    @ApiResponse(
        responseCode = "404",
        description = "Topic was not found",
        content =
            @Content(
                schema = @Schema(implementation = ErrorDetailResponse.class),
                examples = {
                  @ExampleObject(
                      name = "TOPIC_DOES_NOT_EXIST",
                      description = "Topic with kid does not exist.",
                      value =
                          """
    {
      "correlationId":"f99d9a88-589e-40aa-90fc-ddf75a9bb478",
      "serviceId": "kkmessage-service",
      "faultCode": "TOPIC_DOES_NOT_EXIST",
      "faultTopic": "Topic with kid '%s' does not exist.",
      "faultTopicParams": [
         "fd6c405f-7ae5-4d97-816e-a929c04b7001"
      ]
    }
    """)
                }))
  })
  @PostMapping(value = TOPICS_URI)
  public ResponseEntity<?> create(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of organization.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_ORGANIZATION_ID)
          UUID organizationId,
      @Valid @RequestBody TopicCreateRequest request) {
    log.info("Calling TopicController POST create: [{}]; request ({})", TOPICS_URI, request);

    return new ResponseEntity<>(
        TopicResponseMapper.map(
            createTopicUseCase.execute(
                createTopicCommand(
                    request, organizationId, getStringClaim(jwt, EMPLOYEE_ID_CLAIM_NAME)))),
        HttpStatus.CREATED);
  }

  @Operation(
      summary = "Updates a topic.",
      description = "Updates a topic.",
      requestBody =
          @io.swagger.v3.oas.annotations.parameters.RequestBody(
              description = "Body of request for updated topic.",
              required = true,
              content = @Content(schema = @Schema(implementation = TopicUpdateRequest.class))),
      extensions = {
        @Extension(
            properties = {
              @ExtensionProperty(name = "x-eventcatalog-message-name", value = "Topic - Update")
            })
      })
  @ApiResponses({
    @ApiResponse(
        responseCode = "201",
        description = "Successful operation - response with details of the updated topic.",
        content = @Content(schema = @Schema(implementation = TopicDetailResponse.class))),
    @ApiResponse(
        responseCode = "404",
        description = "Topic was not found",
        content =
            @Content(
                schema = @Schema(implementation = ErrorDetailResponse.class),
                examples = {
                  @ExampleObject(
                      name = "TOPIC_DOES_NOT_EXIST",
                      description = "Topic with kid does not exist.",
                      value =
                          """
{
"correlationId":"f99d9a88-589e-40aa-90fc-ddf75a9bb478",
"serviceId": "kkmessage-service",
"faultCode": "TOPIC_DOES_NOT_EXIST",
"faultTopic": "Topic with kid '%s' does not exist.",
"faultTopicParams": [
"fd6c405f-7ae5-4d97-816e-a929c04b7001"
]
}
""")
                }))
  })
  @PutMapping(value = TOPIC_URI)
  public ResponseEntity<?> update(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of organization.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_ORGANIZATION_ID)
          UUID organizationId,
      @Parameter(
              description = "Unique identifier of organization.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_TOPIC_ID)
          UUID topicId,
      @Valid @RequestBody TopicUpdateRequest request) {
    log.info("Calling TopicController PUT update: [{}]; request ({})", TOPICS_URI, request);

    return new ResponseEntity<>(
        TopicResponseMapper.map(
            updateTopicUseCase.execute(
                updateTopicCommand(
                    request,
                    topicId,
                    organizationId,
                    getStringClaim(jwt, EMPLOYEE_ID_CLAIM_NAME)))),
        HttpStatus.OK);
  }

  @Operation(
      summary = "Deletes a topic.",
      description = "Deletes a topic.",
      extensions = {
        @Extension(
            properties = {
              @ExtensionProperty(name = "x-eventcatalog-message-name", value = "Topic - Delete")
            })
      })
  @ApiResponses({
    @ApiResponse(responseCode = "204", description = "Successful operation - topic deleted."),
    @ApiResponse(
        responseCode = "404",
        description = "Topic was not found",
        content =
            @Content(
                schema = @Schema(implementation = ErrorDetailResponse.class),
                examples = {
                  @ExampleObject(
                      name = "TOPIC_DOES_NOT_EXIST",
                      description = "Topic with kid does not exist.",
                      value =
                          """
{
"correlationId":"f99d9a88-589e-40aa-90fc-ddf75a9bb478",
"serviceId": "kkmessage-service",
"faultCode": "TOPIC_DOES_NOT_EXIST",
"faultTopic": "Topic with kid '%s' does not exist.",
"faultTopicParams": [
"fd6c405f-7ae5-4d97-816e-a929c04b7001"
]
}
""")
                }))
  })
  @DeleteMapping(value = TOPIC_URI)
  public ResponseEntity<?> delete(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of organization.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_ORGANIZATION_ID)
          UUID organizationId,
      @Parameter(
              description = "Unique identifier of organization.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_TOPIC_ID)
          UUID topicId) {
    log.info("Calling TopicController PUT update: [{}];", TOPICS_URI);
    deleteTopicUseCase.execute(
        deleteTopicCommand(topicId, organizationId, getStringClaim(jwt, EMPLOYEE_ID_CLAIM_NAME)));

    return ResponseEntity.noContent().build();
  }
}
