package sk.kosice.konto.kkmessageservice.message;

import static org.assertj.core.api.Assertions.assertThat;
import static sk.kosice.konto.kkmessageservice.message.MessageApiFixtures.createMessageRequest;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.UNAUTHORIZED;
import static sk.kosice.konto.kkmessageservice.restapi.common.listing.ListOrderingParser.ASCENDING;
import static sk.kosice.konto.kkmessageservice.restapi.common.listing.ListOrderingParser.DESCENDING;

import java.util.List;
import java.util.UUID;
import org.junit.jupiter.api.Test;
import sk.kosice.konto.kkmessageservice.domain.common.listing.MessageListingAttribute;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.BaseMessageDto;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageListResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageWithKidListResponse;

public class MessageListingFeatureSpec extends MessageFeatureSpec {

  @Test
  public void thatEmptyMessageListCanBeRetrieved() {
    final MessageListResponse messages =
        listMessagesByRecipientKid(UUID.fromString(EXTENSION_KID_CLAIM));
    assertThat(messages.items()).isEmpty();
    assertThat(messages.totalCount()).isZero();
  }

  @Test
  public void thatMessageListCanBeRetrievedByRecipientKid() {
    createMessagesForFilteringAndSorting(EXTENSION_KID_CLAIM, ORGANIZATION_ID_1);

    final MessageListResponse messages =
        listMessagesByRecipientKid(UUID.fromString(EXTENSION_KID_CLAIM));
    assertThat(messages.items()).isNotEmpty();
    assertThat(messages.totalCount()).isEqualTo(3);
  }

  @Test
  public void thatMessageListCanBeRetrievedByOrganizationId() {
    final UUID organizationId = ORGANIZATION_ID_2;
    createMessagesForFilteringAndSorting(EXTENSION_KID_CLAIM, organizationId);

    final MessageWithKidListResponse messages = listMessagesByOrganizationId(organizationId);
    assertThat(messages.items()).isNotEmpty();
    assertThat(messages.totalCount()).isEqualTo(2);
  }

  @Test
  public void thatMessageListCantBeRetrievedByOrganizationIdWithoutEmployeeToken() {
    final UUID organizationId = ORGANIZATION_ID_1;
    createMessagesForFilteringAndSorting(EXTENSION_KID_CLAIM, organizationId);

    final ErrorDetailResponse errorDetailResponse =
        listMessagesByOrganizationIdWithoutJwt(organizationId);
    assertThat(errorDetailResponse.primaryFault().get().faultCode())
        .isNotNull()
        .isEqualTo(UNAUTHORIZED.toString());
  }

  @Test
  public void thatMessageListCanBeFilteredByTitle() {
    final List<BaseMessageDto> messages =
        createMessagesForFilteringAndSorting(EXTENSION_KID_CLAIM, ORGANIZATION_ID_1);

    final String filter =
        String.format(
            "?filter=%s==%s", MessageListingAttribute.TITLE.apiName(), messages.get(0).getTitle());

    checkFiltering(UUID.fromString(EXTENSION_KID_CLAIM), filter, 1, messages.get(0));
  }

  @Test
  public void thatMessageListCanBeSortedByCreatedAt() {
    final List<BaseMessageDto> messages =
        createMessagesForFilteringAndSorting(EXTENSION_KID_CLAIM, ORGANIZATION_ID_1);

    checkSorting(
        UUID.fromString(EXTENSION_KID_CLAIM),
        MessageListingAttribute.CREATED_AT.apiName() + ASCENDING,
        3,
        messages.get(0),
        messages.get(1),
        messages.get(2));

    checkSorting(
        UUID.fromString(EXTENSION_KID_CLAIM),
        MessageListingAttribute.CREATED_AT.apiName() + DESCENDING,
        3,
        messages.get(2),
        messages.get(1),
        messages.get(0));
  }

  private List<BaseMessageDto> createMessagesForFilteringAndSorting(
      String recipientKid, UUID organizationId) {
    persistAndFindRecipient(recipientRepository, prepareRecipientData(recipientKid));
    final UUID topicId = persistAndFindTopic(topicRepository, ORGANIZATION_ID_1).id();
    persistAndFindSubscription(
        subscriptionRepository, topicId, UUID.fromString(EXTENSION_KID_CLAIM));

    final MessageDetailResponse message1 =
        createMessage(
            UUID.fromString(recipientKid), createMessageRequest(topicId).withTitle("title1"));

    final MessageDetailResponse message2 =
        createMessage(
            UUID.fromString(recipientKid), createMessageRequest(topicId).withTitle("title2"));

    final MessageDetailResponse message3 =
        createMessage(
            UUID.fromString(recipientKid), createMessageRequest(topicId).withTitle("title3"));

    final MessageDetailResponse message4 =
        createPublicMessage(organizationId, createMessageRequest(topicId).withTitle("title4"));

    final MessageDetailResponse message5 =
        createPublicMessage(organizationId, createMessageRequest(topicId).withTitle("title5"));

    return List.of(
        BaseMessageDto.of(message1),
        BaseMessageDto.of(message2),
        BaseMessageDto.of(message3),
        BaseMessageDto.of(message4),
        BaseMessageDto.of(message5));
  }
}
