'use client';

import {
  ContentLayout,
  Error,
  QueryHandler,
  CreateTopicDialog,
  DeleteTopicDialog,
  TopicsList,
  TopicsListPlaceholder,
  TopicsListSkeleton,
  UpdateTopicDialog,
  withPermission
} from '@/components';

import { useNotifications } from '@/hooks';
import { api, Page, RequestError, Topic, unwrap } from '@/utils';
import { Pagination, PrimaryButton, SearchBar } from '@/lib/idsk';
import { useQuery } from '@tanstack/react-query';
import axios from 'axios';
import { useSession } from 'next-auth/react';
import { useRef, useState } from 'react';

const PAGE_SIZE = 7;

type TopicSort = 'name;' | 'name:';

const ThemesPage = () => {
  const [page, setPage] = useState<number>(1);
  const [sort, setSort] = useState<TopicSort>('name;');
  const [search, setSearch] = useState<string>('');

  const [topicToDelete, setTopicToDelete] = useState<Topic | null>(null);
  const [topicToUpdate, setTopicToUpdate] = useState<Topic | null>(null);

  const ref = useRef<HTMLInputElement>(null);

  const notify = useNotifications();

  const { data } = useSession();
  const organizationId = data?.user.organization?.id as string;

  const changeSort = () => {
    if (sort.endsWith(':')) {
      setSort('name;');
    } else {
      setSort('name:');
    }
  };

  const query = useQuery<Page<Topic>, RequestError>({
    queryKey: ['topics', page, sort, search, PAGE_SIZE],
    queryFn: () =>
      unwrap(
        axios.get(`/api${api.topics(organizationId)}`, {
          params: {
            page,
            'page-size': PAGE_SIZE,
            sort,
            filter: `name=like='*${search}*',description=like='*${search}*'`
          }
        })
      )
  });

  const handleCreate = () => {
    notify('Téma bola úspešne vytvorená.');
    query.refetch();
  };

  const handleUpdate = () => {
    setTopicToUpdate(null);
    notify('Téma bola úspešne aktualizovaná.');
    query.refetch();
  };

  const handleDelete = (data: Page<Topic>) => {
    setTopicToDelete(null);
    notify('Téma bola úspešne odstránená.');

    if (data.page > 1 && data.items.length === 1) {
      setPage((p) => p - 1);
      return;
    }

    query.refetch();
  };

  const onSearch = () => {
    const searchbar = ref.current;
    if (searchbar) {
      setSearch(searchbar.value);
    }
  };

  const onClearSearch = () => {
    setSearch('');
  };

  return (
    <ContentLayout
      title="Témy"
      actions={
        <>
          <SearchBar
            ref={ref}
            className="w-full tb1:w-auto"
            searchbarSize="medium"
            showCancelButton={!!search}
            buttonOnClick={onSearch}
            onCancel={onClearSearch}
            fullWidth
          />
          <CreateTopicDialog onCreate={handleCreate} organizationId={organizationId}>
            <PrimaryButton className="ml-auto w-full tb1:w-auto">Vytvoriť tému</PrimaryButton>
          </CreateTopicDialog>
        </>
      }
    >
      <QueryHandler
        query={query}
        error={
          <TopicsListPlaceholder>
            <Error err={query.error} transparent />
          </TopicsListPlaceholder>
        }
        loading={<TopicsListSkeleton pageSize={PAGE_SIZE} />}
      >
        {(data: Page<Topic>) => (
          <div>
            <TopicsList
              data={data}
              sort={sort.endsWith(':') ? 'asc' : 'desc'}
              onSortChange={changeSort}
              onUpdateClick={(item: Topic) => setTopicToUpdate(item)}
              onDeleteClick={(item: Topic) => setTopicToDelete(item)}
            />
            {data.totalCount > PAGE_SIZE && (
              <div className="mt-7 w-full flex justify-end">
                <Pagination
                  forcePage={page}
                  pageCount={Math.ceil(data.totalCount / PAGE_SIZE)}
                  caption={''}
                  onPageChange={({ selected }) => setPage(selected)}
                />
              </div>
            )}
            <DeleteTopicDialog
              topic={topicToDelete}
              organizationId={organizationId}
              onDelete={() => handleDelete(data)}
              onClose={() => setTopicToDelete(null)}
            />

            <UpdateTopicDialog
              topic={topicToUpdate}
              organizationId={organizationId}
              onUpdate={handleUpdate}
              onClose={() => setTopicToUpdate(null)}
            />
          </div>
        )}
      </QueryHandler>
    </ContentLayout>
  );
};

export default withPermission(ThemesPage, 'ACTION_MANAGE_TOPIC');
