'use client';

import React, { useEffect, useId, useRef, useState } from 'react';
import { useTagTheme } from '@/hooks';
import { cn, COLORS } from '@/utils';
import { CloseIcon, ColorLensIcon, useClickOutside, BaseButton } from '@/lib/idsk';
import { v4 as uuidv4 } from 'uuid';

type CustomTag = {
  id: string;
  value: string;
  color: string;
};

interface TagsInputProps extends React.InputHTMLAttributes<HTMLInputElement> {
  label?: string;
  labelSrOnly?: boolean;
  optionalText?: string;
  mandatory?: boolean;
  subtitle?: string;
  caption?: string;
  onTagsChange: (tags: CustomTag[]) => void;
}

interface TagsInputItemProps {
  item: CustomTag;
  onRemove: () => void;
  onChangeColor: (color: string) => void;
}

const TagsInputItem: React.FC<TagsInputItemProps> = ({ item, onRemove, onChangeColor }) => {
  const [colorListOpened, setColorListOpened] = useState(false);
  const { border, background, text } = useTagTheme(item.color);

  const dropdownRef = useRef<HTMLDivElement>(null);
  useClickOutside(() => setColorListOpened(false), dropdownRef);

  const changeColor = (newColor: string) => {
    onChangeColor(newColor);
    setColorListOpened(false);
  };

  return (
    <div
      className="idsk-tag idsk-tag--small whitespace-nowrap relative"
      style={{ backgroundColor: background, borderColor: border, color: text }}
    >
      <BaseButton ariaLabel="Zmeniť farbu" type="button" onClick={() => setColorListOpened((p) => !p)}>
        <ColorLensIcon />
      </BaseButton>
      {item.value}
      <BaseButton ariaLabel={`Odstranit štítok ${item.value}`} type="button" onClick={onRemove}>
        <CloseIcon />
      </BaseButton>

      {colorListOpened && (
        <div
          ref={dropdownRef}
          className="absolute top-full mt-2 left-0 z-40 rounded-md shadow-medium idsk-text-body max-h-64 overflow-y-auto bg-white"
        >
          {COLORS.map((color) => (
            <BaseButton
              key={`${color.name}-${color.value}`}
              className="px-5 py-3 hover:bg-primary-light text-left flex items-center gap-4 w-full"
              type="button"
              onClick={() => changeColor(color.value)}
            >
              <div className="w-4 h-4 rounded" style={{ backgroundColor: color.value }} />
              {color.name}
            </BaseButton>
          ))}
        </div>
      )}
    </div>
  );
};

const TagsInput: React.FC<TagsInputProps> = ({
  label,
  labelSrOnly,
  optionalText,
  mandatory,
  subtitle,
  caption,
  onTagsChange,
  ...props
}) => {
  const [tags, setTags] = useState<CustomTag[]>([]);
  const [isFocused, setIsFocused] = useState(false);

  const inputRef = useRef<HTMLInputElement>(null);
  const useid = useId();
  const id = props.id || useid;

  const handleKeyDown = (event: React.KeyboardEvent<HTMLInputElement>) => {
    if (event.key === 'Enter') {
      event.preventDefault();
      const target = event.target as HTMLInputElement;
      const value = target.value;
      setTags([...tags, { id: uuidv4(), color: COLORS[0].value, value }]);
      target.value = '';
    }
  };

  const handleRemove = (id: string) => {
    setTags((p) => p.filter((item) => item.id !== id));
  };

  const handleChangeColor = (item: CustomTag, color: string) => {
    setTags(tags.map((tag) => (tag.id === item.id ? { ...tag, color } : tag)));
  };

  useEffect(() => {
    onTagsChange(tags);
  }, [tags, onTagsChange]);

  return (
    <div>
      {!!label && (
        <label className={cn('idsk-input__label', { 'sr-only': labelSrOnly })} htmlFor={id}>
          {label}
          {optionalText && <span className="idsk-input__label--optional"> {optionalText}</span>}
          {mandatory && <span className="idsk-input__label--mandatory"> *</span>}
        </label>
      )}
      {!!subtitle && <p className="idsk-input__subtitle">{subtitle}</p>}
      <div
        className={cn(
          'idsk-text-body idsk-input__wrapper idsk-input flex items-center gap-2 p-1.5 flex-wrap idsk-focusable hover:cursor-text',
          {
            'outline outline-[3px] outline-offset-2 outline-focus-state z-10 ': isFocused
          }
        )}
        onClick={() => inputRef.current?.focus()}
      >
        {tags.map((item) => (
          <TagsInputItem
            key={item.id}
            item={item}
            onRemove={() => handleRemove(item.id)}
            onChangeColor={(color: string) => handleChangeColor(item, color)}
          />
        ))}
        <input
          id={id}
          ref={inputRef}
          onKeyDown={handleKeyDown}
          onFocus={() => setIsFocused(true)}
          onBlur={() => setIsFocused(false)}
          className="p-0.5 ml-2 focus-visible:outline-none placeholder:text-neutral-600"
          autoComplete="off"
          enterKeyHint="enter"
          type="text"
          {...props}
        />
      </div>
      {!!caption && (
        <p className="idsk-input__caption">
          <span>{caption}</span>
        </p>
      )}
    </div>
  );
};

export default TagsInput;
