package sk.kosice.konto.kkmessageservice.repository.message;

import static org.jooq.impl.DSL.exists;
import static org.jooq.impl.DSL.noCondition;
import static sk.kosice.konto.kkmessageservice.domain.message.error.MessageErrorCode.MESSAGE_DOES_NOT_EXIST;
import static sk.kosice.konto.kkmessageservice.repository.message.mapper.JooqMessageRepositoryMapper.listingMapper;
import static sk.kosice.konto.kkmessageservice.repository.message.mapper.JooqMessageRepositoryMapper.messageRecordMapper;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.MESSAGE;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.SUBSCRIPTION;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.TOPIC;

import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.JSON;
import org.jooq.Record;
import org.jooq.SelectJoinStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import sk.kosice.konto.kkmessageservice.domain.common.error.BusinessException;
import sk.kosice.konto.kkmessageservice.domain.message.entity.ListOfMessages;
import sk.kosice.konto.kkmessageservice.domain.message.entity.MessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.query.FindMessageByIdAndKidQuery;
import sk.kosice.konto.kkmessageservice.domain.message.query.FindMessageByIdAndOrganizationIdQuery;
import sk.kosice.konto.kkmessageservice.domain.message.query.FindMessageByIdQuery;
import sk.kosice.konto.kkmessageservice.domain.message.query.MessageListingQuery;
import sk.kosice.konto.kkmessageservice.repository.JooqRepository;
import sk.kosice.konto.kkmessageservice.repository.message.error.constraint.MessageConstraintErrorCode;
import sk.kosice.konto.kkmessageservice.repository.provider.MessageJooqRsqlMetadataConfigProvider;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.handler.JooqPaginatedSelectQueryHandler;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.handler.mapper.PaginatedResultMapper;

@Repository
public class JooqMessageRepository extends JooqRepository implements MessageRepositoryAdapter {

  @Autowired
  protected JooqMessageRepository(DSLContext dslContext) {
    super(dslContext);
  }

  @Override
  public void insert(MessageEntity data) throws BusinessException {
    log.info("Applying {}", data);

    handleInsertResult(
        () ->
            dslContext
                .insertInto(MESSAGE)
                .set(MESSAGE.ID, data.id())
                .set(MESSAGE.ORGANIZATION_ID, data.organizationId())
                .set(MESSAGE.SENDER_NAME, data.senderName())
                .set(MESSAGE.TITLE, data.title())
                .set(MESSAGE.TOPIC_ID, data.topicId())
                .set(MESSAGE.BODY, data.body())
                .set(MESSAGE.TAGS, data.tags().map(JSON::valueOf).orElse(null))
                .set(MESSAGE.RECIPIENT_KID, data.recipientKid().orElse(null))
                .set(MESSAGE.CREATED_AT, data.createdAt())
                .set(MESSAGE.IS_NOTIFICATION_SEND, data.isNotificationSend())
                .set(MESSAGE.BODY_TYPE, data.bodyType())
                .set(MESSAGE.BODY_SHORT, data.bodyShort())
                .set(MESSAGE.ACTIONS, data.actions().map(JSON::json).orElse(null))
                .set(
                    MESSAGE.LOCATIONS,
                    DSL.jsonArray(data.locations().stream().map(Enum::name).map(DSL::val).toList()))
                .execute(),
        data,
        MessageConstraintErrorCode.values());
  }

  @Override
  public MessageEntity findOne(FindMessageByIdQuery query) throws BusinessException {
    log.info("Applying {}", query);

    return handleFindOneResult(
        () -> {
          Condition condition = noCondition().and(MESSAGE.ID.eq(query.id()));
          if (query.recipientKid().isPresent()) {
            condition = condition.and(MESSAGE.RECIPIENT_KID.eq(query.recipientKid().get()));
          } else {
            condition = condition.and(MESSAGE.RECIPIENT_KID.isNull());
          }
          return dslContext
              .select()
              .from(MESSAGE)
              .join(TOPIC)
              .onKey()
              .where(condition)
              .fetch(messageRecordMapper());
        },
        query,
        MessageConstraintErrorCode.values());
  }

  @Override
  public MessageEntity findOne(FindMessageByIdAndKidQuery query) throws BusinessException {
    log.info("Applying {}", query);

    return handleFindOneResult(
        () ->
            dslContext
                .select()
                .from(MESSAGE)
                .join(TOPIC)
                .onKey()
                .where(
                    MESSAGE
                        .ID
                        .eq(query.id())
                        .and(
                            MESSAGE
                                .RECIPIENT_KID
                                .eq(query.recipientKid())
                                .or(MESSAGE.RECIPIENT_KID.isNull())))
                .fetch(messageRecordMapper()),
        query,
        MessageConstraintErrorCode.values());
  }

  @Override
  public MessageEntity findOne(FindMessageByIdAndOrganizationIdQuery query)
      throws BusinessException {
    log.info("Applying {}", query);

    return handleFindOneResult(
        () ->
            dslContext
                .select()
                .from(MESSAGE)
                .join(TOPIC)
                .onKey()
                .where(
                    MESSAGE
                        .ID
                        .eq(query.id())
                        .and(MESSAGE.ORGANIZATION_ID.eq(query.organizationId())))
                .fetch(messageRecordMapper()),
        query,
        MessageConstraintErrorCode.values());
  }

  @Override
  public ListOfMessages list(MessageListingQuery query) throws BusinessException {
    log.info("Applying {}", query);

    SelectJoinStep<Record> select = dslContext.select().from(MESSAGE).join(TOPIC).onKey();
    Condition condition = noCondition();
    if (query.recipientKid().isPresent()) {
      condition =
          MESSAGE
              .RECIPIENT_KID
              .eq(query.recipientKid().get())
              .or(
                  MESSAGE
                      .RECIPIENT_KID
                      .isNull()
                      .and(
                          exists(
                              dslContext
                                  .select(DSL.val(1))
                                  .from(SUBSCRIPTION)
                                  .where(
                                      SUBSCRIPTION
                                          .TOPIC_ID
                                          .eq(MESSAGE.TOPIC_ID)
                                          .and(
                                              SUBSCRIPTION.RECIPIENT_KID.eq(
                                                  query.recipientKid().get()))
                                          .and(SUBSCRIPTION.IS_EMAIL_ENABLED.isTrue())))));
    }

    if (query.organizationId().isPresent()) {
      condition = MESSAGE.ORGANIZATION_ID.eq(query.organizationId().get());
    }

    select.where(condition);

    final JooqPaginatedSelectQueryHandler<MessageListingQuery, ListOfMessages>
        paginatedListingHandler =
            new JooqPaginatedSelectQueryHandler<>(
                query, new MessageJooqRsqlMetadataConfigProvider());

    final PaginatedResultMapper<MessageListingQuery, ListOfMessages> paginated =
        paginatedListingHandler.handle(select.getQuery());

    return paginated.map(listingMapper);
  }

  @Override
  protected void createEntityDoesNotExistError(Object object) {
    switch (object) {
      case FindMessageByIdQuery query ->
          throw MESSAGE_DOES_NOT_EXIST.createError(query.id()).convertToException();
      case FindMessageByIdAndKidQuery query ->
          throw MESSAGE_DOES_NOT_EXIST.createError(query.id()).convertToException();
      case FindMessageByIdAndOrganizationIdQuery query ->
          throw MESSAGE_DOES_NOT_EXIST.createError(query.id()).convertToException();
      default -> throw MESSAGE_DOES_NOT_EXIST.createError(object).convertToException();
    }
  }
}
