package sk.kosice.konto.kkmessageservice.business.topic.service;

import static sk.kosice.konto.kkmessageservice.domain.common.enumeration.Permission.ACTION_MANAGE_TOPIC;

import jakarta.inject.Inject;
import jakarta.inject.Named;
import java.util.UUID;
import sk.kosice.konto.kkmessageservice.business.AbstractService;
import sk.kosice.konto.kkmessageservice.business.organization.port.outbound.QueryOrganizationPort;
import sk.kosice.konto.kkmessageservice.business.permission.port.outbound.CheckPermissionsUseCase;
import sk.kosice.konto.kkmessageservice.business.subscription.port.outbound.PersistSubscriptionPort;
import sk.kosice.konto.kkmessageservice.business.topic.port.inbound.CreateTopicUseCase;
import sk.kosice.konto.kkmessageservice.business.topic.port.outbound.PersistTopicPort;
import sk.kosice.konto.kkmessageservice.business.topic.port.outbound.QueryTopicPort;
import sk.kosice.konto.kkmessageservice.domain.common.transactional.Transactional;
import sk.kosice.konto.kkmessageservice.domain.topic.command.CreateTopicCommand;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.ImmutableTopicEntity;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.TopicEntity;
import sk.kosice.konto.kkmessageservice.domain.topic.query.FindTopicByIdQuery;

@Named
public class CreateTopicService extends AbstractService implements CreateTopicUseCase {

  private final PersistTopicPort persistTopicPort;
  private final QueryTopicPort queryTopicPort;
  private final PersistSubscriptionPort persistSubscriptionPort;
  private final Transactional transactional;
  private final QueryOrganizationPort queryOrganizationPort;
  private final CheckPermissionsUseCase checkPermissionsUseCase;

  @Inject
  public CreateTopicService(
      PersistTopicPort persistTopicPort,
      QueryTopicPort queryTopicPort,
      PersistSubscriptionPort persistSubscriptionPort,
      Transactional transactional,
      QueryOrganizationPort queryOrganizationPort,
      CheckPermissionsUseCase checkPermissionsUseCase) {
    this.persistTopicPort = persistTopicPort;
    this.queryTopicPort = queryTopicPort;
    this.persistSubscriptionPort = persistSubscriptionPort;
    this.transactional = transactional;
    this.queryOrganizationPort = queryOrganizationPort;
    this.checkPermissionsUseCase = checkPermissionsUseCase;
  }

  @Override
  public TopicEntity execute(CreateTopicCommand command) {
    checkPermission(
        checkPermissionsUseCase, command.actorId(), ACTION_MANAGE_TOPIC, command.organizationId());

    final var organization = queryOrganizationPort.findOne(command.organizationId());

    final TopicEntity data =
        ImmutableTopicEntity.builder()
            .id(UUID.randomUUID())
            .name(command.name())
            .description(command.description())
            .organizationId(organization.id())
            .organizationName(organization.name())
            .build();

    return transactional.result(
        () -> {
          persistTopicPort.insert(data);
          final var entity = queryTopicPort.findOne(FindTopicByIdQuery.of(data));

          persistSubscriptionPort.createNewSubscriptionForEachRecipient(entity.id());
          return entity;
        });
  }
}
