import NextAuth from 'next-auth';
import MicrosoftEntraId from 'next-auth/providers/microsoft-entra-id';
import CredentialsProvider from 'next-auth/providers/credentials';
import { getProfile } from './utils/auth/server';
import { isExpired, IUser, refreshToken } from './utils';
import { jwtDecode } from 'jwt-decode';
import { signInConfig, signUpConfig } from './lib/auth';

export const { handlers, auth, signIn, signOut } = NextAuth({
  providers: [
    CredentialsProvider({
      id: 'credentials',
      name: 'Credentials',
      credentials: {
        id_token: { label: 'id_token', type: 'text' },
        access_token: { label: 'access_token', type: 'text' },
        refresh_token: { label: 'refresh_token', type: 'text' },
        expires_on: { label: 'expires_on', type: 'text' }
      },
      async authorize({ id_token, access_token, refresh_token, expires_on }) {
        const decoded = jwtDecode(id_token as string) as any;
        return {
          id: '',
          name: '',
          kid: decoded.extension_kid,
          id_token,
          access_token,
          refresh_token,
          expires_on: Number(expires_on)
        };
      }
    }),
    MicrosoftEntraId(signInConfig),
    MicrosoftEntraId(signUpConfig)
  ],
  callbacks: {
    jwt: async ({ user, token, account }) => {
      const typedToken = token as any;

      if (account) {
        let tokens: any = {};
        if (account.type === 'credentials') {
          const credentialsUser = user as any;
          tokens.id_token = credentialsUser.id_token;
          tokens.access_token = credentialsUser.access_token;
          tokens.refresh_token = credentialsUser.refresh_token;
          tokens.expires_on = credentialsUser.expires_on;
        } else {
          tokens = account;
        }

        const { data, error } = await getProfile((user as any).kid, tokens.access_token ?? '');

        if (error) {
          typedToken.error = error;
          return typedToken;
        }

        if (data) {
          const { identity } = data;
          typedToken.user = {
            id: identity.kid,
            name: identity.givenName,
            surname: identity.surname,
            email: identity.emailPreferred,
            created: identity.created,
            modified: identity.modified,
            pin: identity.pin
          } as IUser;
        }

        typedToken.accessToken = tokens.access_token;
        typedToken.refreshToken = tokens.refresh_token;
        typedToken.expiresOn = tokens.expires_on;
      }

      if (isExpired(typedToken.expiresOn)) {
        try {
          const refreshedToken = await refreshToken(typedToken);
          return refreshedToken;
        } catch (_) {
          typedToken.error = 'RefreshTokenError';
        }
      }

      return typedToken;
    },

    session: async ({ session, token }) => {
      const sameSession = session as any;
      const sameToken = token as any;

      if (sameToken.user) {
        sameSession.user = sameToken.user;
      }

      sameSession.idToken = sameToken.idToken;
      sameSession.accessToken = sameToken.accessToken;
      sameSession.refreshToken = sameToken.refreshToken;
      sameSession.expiresOn = sameToken.expiresOn;
      sameSession.error = sameToken.error;

      return session;
    }
  },
  pages: {
    error: '/error',
    signIn: '/signin'
  },
  session: { strategy: 'jwt' }
});
