'use client';

import { useMutation, useQueryClient } from '@tanstack/react-query';
import { useTranslations } from 'next-intl';
import React, { useRef } from 'react';
import axios from 'axios';
import { useSession } from 'next-auth/react';
import { Field, Form, Formik } from 'formik';
import { z } from 'zod';
import { toFormikValidationSchema } from 'zod-formik-adapter';
import { Dialog, Input, PrimaryButton, SecondaryButton } from '@/lib/idsk';
import { unwrap } from '@/utils';

const validationSchema = z.object({
  street: z.string().min(1, { message: 'validation_required' }),
  census: z.string({ invalid_type_error: 'validation_number' }).min(1, { message: 'validation_required' }),
  postalCode: z.string({ invalid_type_error: 'validation_number' }).min(1, { message: 'validation_required' }),
  city: z.string().min(1, { message: 'validation_required' }),
  country: z.string().min(1, { message: 'validation_required' })
});

interface AddAddressDialogProps {
  type: string | null;
  toggleOpened: () => void;
}

const AddAddressDialog: React.FC<AddAddressDialogProps> = ({ type, toggleOpened }) => {
  const t = useTranslations('my_data');
  const client = useQueryClient();

  const formRef = useRef<HTMLFormElement>(null);

  const { data } = useSession();

  const mutation = useMutation({
    mutationKey: ['address'],
    mutationFn: (mutationData: any) => unwrap(axios.post(`/api/v1/kk/profiles/${data?.user.id}/address`, mutationData)),
    onSuccess: () => {
      client.invalidateQueries({ queryKey: ['profile'] });
      toggleOpened();
    }
  });

  const handleSubmit = () => {
    const form = formRef.current;
    if (form) form.requestSubmit();
  };

  const formAction = (values: any) => {
    const { street, census, postalCode, city, country } = values;
    const address = {
      type: type as string,
      street1: street,
      city,
      state: country,
      postalCode,
      street2: '',
      censusNumber: census,
      orientationNumber: '',
      country: '',
      floor: '',
      flat: '',
      latitude: 0,
      longitude: 0
    };
    mutation.mutate(address);
  };

  return (
    <Dialog
      title={type ? t(`${type}_address`) : ''}
      opened={!!type}
      toggleOpened={toggleOpened}
      primaryButton={
        <PrimaryButton type="button" disabled={mutation.status === 'pending'} onClick={handleSubmit}>
          {t('save')}
        </PrimaryButton>
      }
      secondaryButton={<SecondaryButton onClick={toggleOpened}>{t('cancel')}</SecondaryButton>}
    >
      <Formik
        initialValues={{
          street: '',
          census: null,
          postalCode: null,
          city: '',
          country: ''
        }}
        validationSchema={toFormikValidationSchema(validationSchema)}
        validateOnMount={false}
        validateOnBlur={false}
        validateOnChange={false}
        onSubmit={(values) => formAction(values)}
      >
        {({ errors }) => (
          <Form ref={formRef} className="w-full">
            <Field
              as={Input}
              label={t('street')}
              name="street"
              fullWidth
              required
              mandatory
              error={!!errors.street}
              errorMsg={t(errors.street)}
            />

            <div className="flex flex-col tb2:flex-row tb2:gap-5 w-full">
              <div>
                <Field
                  as={Input}
                  label={t('consensus_number')}
                  name="census"
                  error={!!errors.census}
                  errorMsg={t(errors.census)}
                  fullWidth
                  required
                  mandatory
                />
              </div>
              <div>
                <Field
                  as={Input}
                  label={t('postal_code')}
                  name="postalCode"
                  fullWidth
                  required
                  mandatory
                  error={!!errors.postalCode}
                  errorMsg={t(errors.postalCode)}
                />
              </div>
            </div>

            <Field
              as={Input}
              label={t('city')}
              name="city"
              error={!!errors.city}
              errorMsg={errors.city}
              fullWidth
              required
              mandatory
            />
            <Field
              as={Input}
              label={t('country')}
              name="country"
              error={!!errors.country}
              errorMsg={errors.country}
              fullWidth
              required
              mandatory
            />
          </Form>
        )}
      </Formik>
    </Dialog>
  );
};

export default AddAddressDialog;
