package sk.kosice.konto.kkmessageservice.repository.subscription;

import static org.jooq.impl.DSL.trueCondition;
import static sk.kosice.konto.kkmessageservice.domain.subscription.error.SubscriptionErrorCode.SUBSCRIPTION_DOES_NOT_EXIST;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.*;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.SUBSCRIPTION;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.TOPIC;
import static sk.kosice.konto.kkmessageservice.repository.subscription.mapper.JooqSubscriptionRepositoryMapper.listingMapper;

import cz.jirutka.rsql.parser.RSQLParserException;
import cz.jirutka.rsql.parser.ast.Node;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import sk.kosice.konto.kkmessageservice.domain.subscription.data.DeleteSubscriptionsByOrganizationIdAndKidData;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.BaseSubscriptionEntity;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.ListOfSubscriptions;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.SubscriptionEntity;
import sk.kosice.konto.kkmessageservice.domain.subscription.query.FindSubscribedOrganizationsByKidQuery;
import sk.kosice.konto.kkmessageservice.domain.subscription.query.FindSubscriptionByIdQuery;
import sk.kosice.konto.kkmessageservice.domain.subscription.query.SubscriptionListingQuery;
import sk.kosice.konto.kkmessageservice.repository.JooqRepository;
import sk.kosice.konto.kkmessageservice.repository.model.Keys;
import sk.kosice.konto.kkmessageservice.repository.provider.SubscriptionJooqRsqlMetadataConfigProvider;
import sk.kosice.konto.kkmessageservice.repository.provider.TopicJooqRsqlMetadataConfigProvider;
import sk.kosice.konto.kkmessageservice.repository.rsql.RsqlErrorCode;
import sk.kosice.konto.kkmessageservice.repository.rsql.converter.BooleanValueConverter;
import sk.kosice.konto.kkmessageservice.repository.rsql.converter.DateTimeValueConverter;
import sk.kosice.konto.kkmessageservice.repository.rsql.converter.LocalDateValueConverter;
import sk.kosice.konto.kkmessageservice.repository.rsql.converter.UUIDValueConverter;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.builder.JooqRsqlConditionBuilder;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.handler.JooqPaginatedSelectQueryHandler;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.handler.mapper.PaginatedResultMapper;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata.JooqRsqlMetadataImpl;
import sk.kosice.konto.kkmessageservice.repository.subscription.error.constraint.SubscriptionConstraintErrorCode;
import sk.kosice.konto.kkmessageservice.repository.subscription.mapper.JooqSubscriptionRepositoryMapper;

@Repository
public class JooqSubscriptionRepository extends JooqRepository
    implements SubscriptionRepositoryAdapter {

  @Autowired
  protected JooqSubscriptionRepository(DSLContext dslContext) {
    super(dslContext);
  }

  @Override
  protected void createEntityDoesNotExistError(Object query) {
    switch (query) {
      case FindSubscriptionByIdQuery queryById:
        throw SUBSCRIPTION_DOES_NOT_EXIST.createError(queryById.id()).convertToException();
      default:
        throw SUBSCRIPTION_DOES_NOT_EXIST.createError(query).convertToException();
    }
  }

  @Transactional
  @Override
  public void batchUpdate(List<BaseSubscriptionEntity> entities) {
    log.info("Applying batch update{}", entities);

    handleBatchUpdateResults(
        () ->
            dslContext
                .batch(
                    entities.stream()
                        .map(
                            entity ->
                                dslContext
                                    .update(SUBSCRIPTION)
                                    .set(SUBSCRIPTION.IS_EMAIL_ENABLED, entity.isEmailEnabled())
                                    .where(SUBSCRIPTION.ID.eq(entity.id())))
                        .toList())
                .execute(),
        entities,
        SubscriptionConstraintErrorCode.values());
  }

  @Override
  public void update(BaseSubscriptionEntity entity) {
    log.info("Applying update {}", entity);

    handleUpdateResult(
        () ->
            dslContext
                .update(SUBSCRIPTION)
                .set(SUBSCRIPTION.IS_EMAIL_ENABLED, entity.isEmailEnabled())
                .where(SUBSCRIPTION.ID.eq(entity.id()))
                .execute(),
        entity,
        SubscriptionConstraintErrorCode.values());
  }

  @Override
  public void initSubscriptionsForRecipient(UUID recipientKid) {
    log.info("Applying init for recipient {}", recipientKid);

    dslContext
        .insertInto(SUBSCRIPTION)
        .select(
            dslContext
                .select(DSL.uuid(), TOPIC.ID, DSL.val(recipientKid), DSL.val(false))
                .from(TOPIC))
        .execute();
  }

  @Override
  public void createNewSubscriptionForEachRecipient(UUID topicId) {
    log.info("Applying create new subscription for topic {}", topicId);

    dslContext
        .insertInto(SUBSCRIPTION)
        .columns(
            SUBSCRIPTION.ID,
            SUBSCRIPTION.TOPIC_ID,
            SUBSCRIPTION.RECIPIENT_KID,
            SUBSCRIPTION.IS_EMAIL_ENABLED)
        .select(
            dslContext
                .select(DSL.uuid(), DSL.val(topicId), RECIPIENT.KID, DSL.val(false))
                .from(RECIPIENT))
        .execute();
  }

  @Override
  public void insertSubscriptionsByOrganizationIdAndKid(
      UUID kid, Collection<UUID> organizationIds) {
    log.info(
        "Applying insert subscriptions for recipient with KID='{}' by organizations='{}' ",
        kid,
        organizationIds);

    dslContext
        .insertInto(SUBSCRIPTION)
        .columns(
            SUBSCRIPTION.ID,
            SUBSCRIPTION.TOPIC_ID,
            SUBSCRIPTION.RECIPIENT_KID,
            SUBSCRIPTION.IS_EMAIL_ENABLED)
        .select(
            dslContext
                .select(DSL.uuid(), TOPIC.ID, DSL.val(kid), DSL.val(true))
                .from(TOPIC)
                .where(TOPIC.ORGANIZATION_ID.in(organizationIds)))
        .onConflictOnConstraint(Keys.UNIQUE_SUBSCRIPTION_TOPIC_ID_RECIPIENT_KID)
        .doNothing()
        .execute();
  }

  @Override
  public void batchDeleteByOrganizationIdAndKid(
      DeleteSubscriptionsByOrganizationIdAndKidData data) {
    log.info(
        "Applying delete subscriptions for recipient {} with rsql query {}",
        data.recipientKid(),
        data.rsqlQuery());
    buildConditionFromRsql(data.rsqlQuery());

    dslContext
        .deleteFrom(SUBSCRIPTION)
        .where(SUBSCRIPTION.RECIPIENT_KID.eq(data.recipientKid()))
        .and(
            SUBSCRIPTION.TOPIC_ID.in(
                dslContext
                    .select(TOPIC.ID)
                    .from(TOPIC)
                    .where(buildConditionFromRsql(data.rsqlQuery()))))
        .execute();
  }

  @Override
  public void batchDeleteByTopicId(UUID topicId) {
    log.info("Applying delete subscriptions for all recipients by Topic ID '{}'. ", topicId);

    dslContext.deleteFrom(SUBSCRIPTION).where(SUBSCRIPTION.TOPIC_ID.eq(topicId)).execute();
  }

  private Condition buildConditionFromRsql(Optional<Node> rsqlQuery) {
    if (rsqlQuery.isPresent()) {
      log.debug("Building condition from rsqlQuery {}", rsqlQuery.get());
      final var metadataProvider = new TopicJooqRsqlMetadataConfigProvider();
      final var metadata = new JooqRsqlMetadataImpl();
      metadata
          .registerConverter(new BooleanValueConverter())
          .registerConverter(new DateTimeValueConverter())
          .registerConverter(new UUIDValueConverter())
          .registerConverter(new LocalDateValueConverter());
      metadataProvider.configure(metadata);

      final var builder = new JooqRsqlConditionBuilder();
      builder.setJooqRsqlMetadata(metadata);

      try {
        return builder.build(rsqlQuery.get());
      } catch (RSQLParserException e) {
        throw RsqlErrorCode.PARSE_RSQL_EXCEPTION.createError().convertToException();
      }
    }

    return trueCondition();
  }

  @Override
  public void insert(SubscriptionEntity entity) {
    log.info("Applying insert {}", entity);

    handleInsertResult(
        () ->
            dslContext
                .insertInto(SUBSCRIPTION)
                .set(SUBSCRIPTION.ID, entity.id())
                .set(SUBSCRIPTION.RECIPIENT_KID, entity.recipientKid())
                .set(SUBSCRIPTION.TOPIC_ID, entity.topic().id())
                .set(SUBSCRIPTION.IS_EMAIL_ENABLED, entity.isEmailEnabled())
                .execute(),
        entity,
        SubscriptionConstraintErrorCode.values());
  }

  @Override
  public ListOfSubscriptions list(SubscriptionListingQuery query) {
    log.info("Applying {}", query);

    final var select =
        dslContext
            .select()
            .from(SUBSCRIPTION)
            .join(TOPIC)
            .onKey()
            .where(SUBSCRIPTION.RECIPIENT_KID.eq(query.recipientKid()));

    final JooqPaginatedSelectQueryHandler<SubscriptionListingQuery, ListOfSubscriptions>
        paginatedListingHandler =
            new JooqPaginatedSelectQueryHandler<>(
                query, new SubscriptionJooqRsqlMetadataConfigProvider());

    final PaginatedResultMapper<SubscriptionListingQuery, ListOfSubscriptions> paginated =
        paginatedListingHandler.handle(select.getQuery());

    return paginated.map(listingMapper);
  }

  @Override
  public List<SubscriptionEntity> listWithoutPaging(SubscriptionListingQuery query) {
    return dslContext
        .select()
        .from(SUBSCRIPTION)
        .join(TOPIC)
        .onKey()
        .where(SUBSCRIPTION.RECIPIENT_KID.eq(query.recipientKid()))
        .fetch(JooqSubscriptionRepositoryMapper.subscriptionRecordMapper());
  }

  @Override
  public SubscriptionEntity findOne(FindSubscriptionByIdQuery query) {
    log.info("Applying {}", query);

    return handleFindOneResult(
        () ->
            dslContext
                .selectFrom(SUBSCRIPTION)
                .where(SUBSCRIPTION.RECIPIENT_KID.eq(query.recipientKid()))
                .and(SUBSCRIPTION.ID.eq(query.id()))
                .fetch(JooqSubscriptionRepositoryMapper.subscriptionRecordMapper()),
        query,
        SubscriptionConstraintErrorCode.values());
  }

  @Override
  public List<SubscriptionEntity> listUniqueForKid(FindSubscribedOrganizationsByKidQuery query) {
    return dslContext
        .select()
        .distinctOn(TOPIC.ORGANIZATION_ID)
        .from(SUBSCRIPTION)
        .join(TOPIC)
        .onKey()
        .where(SUBSCRIPTION.RECIPIENT_KID.eq(query.recipientKid()))
        .fetch(JooqSubscriptionRepositoryMapper.subscriptionRecordMapper());
  }
}
