package sk.kosice.konto.kkmessageservice.subscription;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static sk.kosice.konto.kkmessageservice.domain.subscription.error.SubscriptionErrorCode.SUBSCRIPTION_DOES_NOT_EXIST;

import java.util.List;
import java.util.UUID;
import org.junit.jupiter.api.Test;
import org.springframework.http.HttpStatus;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionBatchUpdateRequest;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionListResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.subscription.SubscriptionUpdateRequest;

public class SubscriptionUpdateFeatureSpec extends SubscriptionFeatureSpec {

  @Test
  public void thatSubscriptionsCanBeUpdatedInBatch() {
    final List<UUID> topics = prepareTopics();
    prepareRecipient(UUID.fromString(EXTENSION_KID_CLAIM));
    final List<UUID> subscriptions =
        prepareSubscriptions(UUID.fromString(EXTENSION_KID_CLAIM), topics.toArray(UUID[]::new));

    final List<SubscriptionBatchUpdateRequest> requests =
        subscriptions.stream()
            .map(SubscriptionApiFixtures::subscriptionBatchUpdateRequest)
            .toList();
    updateSubscriptionsBatch(UUID.fromString(EXTENSION_KID_CLAIM), requests, HttpStatus.OK);
    final List<SubscriptionDetailResponse> result =
        listSubscriptions(
                HttpStatus.OK, UUID.fromString(EXTENSION_KID_CLAIM), SubscriptionListResponse.class)
            .items();

    assertThat(result).isNotNull().isNotEmpty().hasSize(3);
    result.forEach(
        s -> {
          assertTrue(topics.contains(s.getTopic().id()));
          assertThat(s.getRecipientKid()).isEqualTo(UUID.fromString(EXTENSION_KID_CLAIM));
          assertThat(s.getIsEmailEnabled()).isTrue();
        });
  }

  @Test
  public void thatSubscriptionCanBeUpdated() {
    final List<UUID> topics = prepareTopics();
    prepareRecipient(UUID.fromString(EXTENSION_KID_CLAIM));
    final UUID subscription =
        prepareSubscriptions(UUID.fromString(EXTENSION_KID_CLAIM), topics.toArray(UUID[]::new))
            .getFirst();

    final SubscriptionUpdateRequest request = SubscriptionApiFixtures.subscriptionUpdateRequest();
    updateSubscription(
        UUID.fromString(EXTENSION_KID_CLAIM), subscription, request, HttpStatus.NO_CONTENT);
    final SubscriptionDetailResponse result =
        listSubscriptions(
                HttpStatus.OK, UUID.fromString(EXTENSION_KID_CLAIM), SubscriptionListResponse.class)
            .items()
            .stream()
            .filter(s -> s.getId().equals(subscription))
            .findFirst()
            .get();

    assertThat(result.getIsEmailEnabled()).isTrue();
  }

  @Test
  public void thatNotExistingSubscriptionsCannotBeUpdatedInBatch() {
    final var id = UUID.randomUUID();
    prepareRecipient(UUID.fromString(EXTENSION_KID_CLAIM));
    final var response =
        updateSubscriptionsBatch(
            UUID.fromString(EXTENSION_KID_CLAIM),
            List.of(SubscriptionApiFixtures.subscriptionBatchUpdateRequest(id)),
            HttpStatus.NOT_FOUND,
            ErrorDetailResponse.class);

    assertThat(response).isNotNull();
    assertThat(response.primaryFault().get().faultCode())
        .isEqualTo(SUBSCRIPTION_DOES_NOT_EXIST.toString());
  }

  @Test
  public void thatNotExistingSubscriptionCannotBeUpdated() {
    final var id = UUID.randomUUID();
    prepareRecipient(UUID.fromString(EXTENSION_KID_CLAIM));
    final var response =
        updateSubscription(
            UUID.fromString(EXTENSION_KID_CLAIM),
            id,
            SubscriptionApiFixtures.subscriptionUpdateRequest(),
            HttpStatus.NOT_FOUND,
            ErrorDetailResponse.class);

    assertThat(response).isNotNull();
    assertThat(response.primaryFault().get().faultCode())
        .isEqualTo(SUBSCRIPTION_DOES_NOT_EXIST.toString());
  }
}
