package sk.kosice.konto.kkmessageservice.repository.message.mapper;

import static sk.kosice.konto.kkmessageservice.repository.error.DbErrorCode.DB_ERROR;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;
import org.jooq.JSON;
import org.jooq.Record;
import org.jooq.RecordMapper;
import sk.kosice.konto.kkmessageservice.domain.common.TriFunction;
import sk.kosice.konto.kkmessageservice.domain.common.enumeration.Location;
import sk.kosice.konto.kkmessageservice.domain.message.entity.BaseMessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.entity.ImmutableBaseMessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.entity.ImmutableListOfMessages;
import sk.kosice.konto.kkmessageservice.domain.message.entity.ImmutableMessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.entity.ListOfMessages;
import sk.kosice.konto.kkmessageservice.domain.message.entity.MessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.query.MessageListingQuery;
import sk.kosice.konto.kkmessageservice.repository.model.tables.records.MessageRecord;
import sk.kosice.konto.kkmessageservice.repository.model.tables.records.TopicRecord;

public final class JooqMessageRepositoryMapper {

  public static RecordMapper<Record, MessageEntity> messageRecordMapper() {
    return record -> {
      final MessageRecord messageRecord = record.into(MessageRecord.class);
      final TopicRecord topicRecord = record.into(TopicRecord.class);

      return ImmutableMessageEntity.builder()
          .id(messageRecord.getId())
          .senderName(messageRecord.getSenderName())
          .title(messageRecord.getTitle())
          .topicId(topicRecord.getId())
          .topicName(topicRecord.getName())
          .createdAt(messageRecord.getCreatedAt())
          .organizationId(messageRecord.getOrganizationId())
          .body(messageRecord.getBody())
          .bodyType(messageRecord.getBodyType())
          .bodyShort(messageRecord.getBodyShort())
          .actions(Optional.ofNullable(messageRecord.getActions()).map(JSON::data))
          .isNotificationSend(messageRecord.getIsNotificationSend())
          .tags(Optional.ofNullable(messageRecord.getTags()).map(JSON::data))
          .recipientKid(Optional.ofNullable(messageRecord.getRecipientKid()))
          .addAllLocations(mapLocations(messageRecord.getLocations()))
          .build();
    };
  }

  private static List<Location> mapLocations(JSON locations) {
    final ObjectMapper objectMapper = new ObjectMapper();

    try {
      return objectMapper.readValue(locations.data(), new TypeReference<List<String>>() {}).stream()
          .map(s -> Location.valueOf(s.toUpperCase()))
          .toList();
    } catch (JsonProcessingException e) {
      throw DB_ERROR.createError("Failed convert locations list").convertToException();
    }
  }

  public static RecordMapper<Record, BaseMessageEntity> messageListingRecordMapper() {
    return record -> {
      final MessageRecord messageRecord = record.into(MessageRecord.class);
      final TopicRecord topicRecord = record.into(TopicRecord.class);

      return ImmutableBaseMessageEntity.builder()
          .id(messageRecord.getId())
          .senderName(messageRecord.getSenderName())
          .title(messageRecord.getTitle())
          .topicId(topicRecord.getId())
          .topicName(topicRecord.getName())
          .createdAt(messageRecord.getCreatedAt())
          .organizationId(messageRecord.getOrganizationId())
          .tags(Optional.ofNullable(messageRecord.getTags()).map(JSON::data))
          .recipientKid(Optional.ofNullable(messageRecord.getRecipientKid()))
          .build();
    };
  }

  public static final TriFunction<Stream<Record>, Integer, MessageListingQuery, ListOfMessages>
      listingMapper =
          (records, totalCount, query) ->
              ImmutableListOfMessages.builder()
                  .page(query.page())
                  .pageSize(query.pageSize().value())
                  .totalCount(totalCount)
                  .addAllItems(
                      records.map(record -> messageListingRecordMapper().map(record)).toList())
                  .build();
}
