package sk.kosice.konto.kkmessageservice.repository.rsql.jooq.handler;

import java.util.function.Function;
import java.util.stream.Stream;
import org.jooq.Record;
import org.jooq.SelectQuery;
import org.jooq.impl.DSL;
import sk.kosice.konto.kkmessageservice.domain.common.TriFunction;
import sk.kosice.konto.kkmessageservice.domain.common.listing.ListingQuery;
import sk.kosice.konto.kkmessageservice.domain.common.listing.PaginatedResult;
import sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingAttribute;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.handler.mapper.PaginatedResultMapper;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.metadata.JooqRsqlMetadataConfigProvider;

public class JooqPaginatedSelectQueryHandler<Q extends ListingQuery<?>, R extends PaginatedResult>
    implements JooqSelectQueryHandler<SelectQuery, PaginatedResultMapper<Q, R>> {

  private final Q query;
  private final JooqOneParamSelectQueryHandler<SelectQuery> rsqlQueryHandler;
  private final JooqOneParamSelectQueryHandler<SelectQuery> orderingQueryHandler;
  private final JooqOneParamSelectQueryHandler<SelectQuery> translationQueryHandler;

  public JooqPaginatedSelectQueryHandler(
      Q query, JooqRsqlMetadataConfigProvider jooqListingMetadataConfigProvider) {
    this.query = query;
    this.rsqlQueryHandler =
        new JooqRsqlSelectQueryHandler(query.rsqlQuery(), jooqListingMetadataConfigProvider);
    this.orderingQueryHandler =
        new JooqOrderingSelectQueryHandler<>(
            query.orderings(), jooqListingMetadataConfigProvider, query.languageCodes());
    this.translationQueryHandler = null;
  }

  public JooqPaginatedSelectQueryHandler(
      Q query,
      JooqRsqlMetadataConfigProvider jooqListingMetadataConfigProvider,
      Enum<? extends ListingAttribute>[] attributes) {
    this.query = query;
    this.rsqlQueryHandler =
        new JooqRsqlSelectQueryHandler(query.rsqlQuery(), jooqListingMetadataConfigProvider);
    this.orderingQueryHandler =
        new JooqOrderingSelectQueryHandler<>(
            query.orderings(), jooqListingMetadataConfigProvider, query.languageCodes());
    this.translationQueryHandler =
        new JooqTranslationQueryHandler<>(jooqListingMetadataConfigProvider, attributes, query);
  }

  @Override
  public PaginatedResultMapper<Q, R> handle(SelectQuery dataQuery) {
    final SelectQuery dataQ = rsqlQueryHandler.handle(dataQuery);
    final Integer totalCount = DSL.using(dataQ.configuration()).fetchCount(dataQ);

    final SelectQuery orderedQuery = orderingQueryHandler.handle(dataQ);
    final SelectQuery groupedByQuery =
        translationQueryHandler != null
            ? translationQueryHandler.handle(orderedQuery)
            : orderedQuery;

    groupedByQuery.addLimit(query.limit());
    groupedByQuery.addOffset(query.offset());

    return new PaginatedResultMapper<>() {
      @Override
      public R map(TriFunction<Stream<Record>, Integer, Q, R> fn) {

        try (Stream<Record> orderedStream = groupedByQuery.fetchStream()) {
          return map(selectQuery -> orderedStream, fn);
        }
      }

      @Override
      public <RECORDS> R map(
          Function<SelectQuery, RECORDS> records, TriFunction<RECORDS, Integer, Q, R> mapper) {
        return mapper.apply(records.apply(groupedByQuery), totalCount, query);
      }
    };
  }
}
