package sk.kosice.konto.kkmessageservice.repository.message;

import java.util.List;
import java.util.Optional;
import java.util.UUID;
import sk.kosice.konto.kkmessageservice.domain.common.TimeProvider;
import sk.kosice.konto.kkmessageservice.domain.common.enumeration.BodyType;
import sk.kosice.konto.kkmessageservice.domain.common.enumeration.Location;
import sk.kosice.konto.kkmessageservice.domain.common.listing.ImmutableListOrdering;
import sk.kosice.konto.kkmessageservice.domain.common.listing.ImmutablePageSize;
import sk.kosice.konto.kkmessageservice.domain.common.listing.MessageListingAttribute;
import sk.kosice.konto.kkmessageservice.domain.message.entity.ImmutableMessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.entity.ListOfMessages;
import sk.kosice.konto.kkmessageservice.domain.message.entity.MessageEntity;
import sk.kosice.konto.kkmessageservice.domain.message.query.FindMessageByIdQuery;
import sk.kosice.konto.kkmessageservice.domain.message.query.ImmutableMessageListingQuery;
import sk.kosice.konto.kkmessageservice.domain.message.query.MessageListingQuery;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.BaseOrganization;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.BaseTopic;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.ImmutableSubscriptionEntity;
import sk.kosice.konto.kkmessageservice.domain.subscription.entity.SubscriptionEntity;
import sk.kosice.konto.kkmessageservice.domain.subscription.query.FindSubscriptionByIdQuery;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.ImmutableTopicEntity;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.TopicEntity;
import sk.kosice.konto.kkmessageservice.domain.topic.query.FindTopicByIdQuery;
import sk.kosice.konto.kkmessageservice.repository.subscription.JooqSubscriptionRepository;
import sk.kosice.konto.kkmessageservice.repository.topic.JooqTopicRepository;

public interface JooqMessageRepositoryTestSupport {

  default ImmutableTopicEntity prepareTopicData() {
    return ImmutableTopicEntity.builder()
        .id(UUID.randomUUID())
        .name("name")
        .description("description")
        .organizationId(UUID.randomUUID())
        .organizationName("organizationName")
        .build();
  }

  default ImmutableSubscriptionEntity prepareSubscriptionData(UUID topicId, UUID kid) {
    return ImmutableSubscriptionEntity.builder()
        .id(UUID.randomUUID())
        .isEmailEnabled(true)
        .recipientKid(kid)
        .topic(new BaseTopic(topicId, null, null))
        .organization(new BaseOrganization(null, null))
        .build();
  }

  default TopicEntity persistAndFindTopic(JooqTopicRepository repository) {
    final var data = prepareTopicData();
    repository.insert(data);
    return repository.findOne(FindTopicByIdQuery.of(data));
  }

  default SubscriptionEntity persistAndFindSubscription(
      JooqSubscriptionRepository repository, UUID topicId, UUID kid) {
    final var data = prepareSubscriptionData(topicId, kid);
    repository.insert(data);
    return repository.findOne(FindSubscriptionByIdQuery.of(data));
  }

  default ImmutableMessageEntity prepareMessageData(UUID topicId) {
    return ImmutableMessageEntity.builder()
        .id(UUID.randomUUID())
        .senderName("sender")
        .title("title")
        .topicId(topicId)
        .topicName("none")
        .createdAt(TimeProvider.offsetNow())
        .organizationId(UUID.randomUUID())
        .body("body")
        .bodyType(BodyType.TEXT)
        .bodyShort("bodyShort")
        .isNotificationSend(true)
        .tags("[{\"a\":\"b\"},{\"c\":\"d\"}]")
        .recipientKid(Optional.empty())
        .addAllLocations(List.of(Location.KOSICE_BARCA, Location.KOSICE_STARO_MESTO))
        .build();
  }

  default ImmutableMessageEntity prepareMessageData(UUID topicId, UUID recipientKid) {
    return prepareMessageData(topicId).withRecipientKid(recipientKid);
  }

  default void persistMessage(JooqMessageRepository repository, MessageEntity insertData) {
    repository.insert(insertData);
  }

  default MessageEntity findMessage(JooqMessageRepository repository, FindMessageByIdQuery query) {
    return repository.findOne(query);
  }

  default MessageEntity persistAndFindMessage(
      JooqMessageRepository repository, MessageEntity insertData) {
    persistMessage(repository, insertData);
    return findMessage(repository, FindMessageByIdQuery.of(insertData));
  }

  default ListOfMessages listMessages(
      JooqMessageRepository repository,
      Optional<UUID> recipientKid,
      Optional<UUID> organizationId) {
    final MessageListingQuery query =
        ImmutableMessageListingQuery.builder()
            .pageSize(ImmutablePageSize.of(100))
            .recipientKid(recipientKid)
            .organizationId(organizationId)
            .addAllOrderings(
                List.of(ImmutableListOrdering.of(MessageListingAttribute.CREATED_AT, true)))
            .rsqlQuery(Optional.empty())
            .build();
    return repository.list(query);
  }

  default ListOfMessages listMessagesByRecipientKid(
      JooqMessageRepository repository, UUID recipientKid) {
    return listMessages(repository, Optional.of(recipientKid), Optional.empty());
  }

  default ListOfMessages listMessagesByOrganizationId(
      JooqMessageRepository repository, UUID organizationId) {
    return listMessages(repository, Optional.empty(), Optional.of(organizationId));
  }

  default ListOfMessages listMessages(JooqMessageRepository repository) {
    return listMessages(repository, Optional.of(UUID.randomUUID()), Optional.empty());
  }
}
