package sk.kosice.konto.kkmessageservice.topic;

import static org.assertj.core.api.AssertionsForClassTypes.assertThat;

import io.restassured.http.ContentType;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.IntStream;
import org.springframework.http.HttpStatus;
import sk.kosice.konto.kkmessageservice.ServiceAppFeatureSpec;
import sk.kosice.konto.kkmessageservice.restapi.controller.TopicController;
import sk.kosice.konto.kkmessageservice.restapi.dto.topic.TopicCreateRequest;
import sk.kosice.konto.kkmessageservice.restapi.dto.topic.TopicDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.topic.TopicListResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.topic.TopicUpdateRequest;

public class TopicFeatureSpec extends ServiceAppFeatureSpec implements TopicFeatureSpecTestSupport {

  protected <T> T createTopic(
      UUID organizationId, TopicCreateRequest request, HttpStatus httpStatus, Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(EMPLOYEE_JWT_TOKEN)
        .body(request)
        .post(TopicController.TOPICS_URI, organizationId)
        .then()
        .log()
        .all()
        .statusCode(httpStatus.value())
        .contentType(ContentType.JSON)
        .extract()
        .body()
        .as(response);
  }

  protected TopicDetailResponse createTopic(UUID organizationId, TopicCreateRequest request) {
    return createTopic(organizationId, request, HttpStatus.CREATED, TopicDetailResponse.class);
  }

  protected <T> T createTopicWithoutToken(
      UUID organizationId, TopicCreateRequest request, HttpStatus httpStatus, Class<T> response) {
    return requestSpecification()
        .when()
        .body(request)
        .post(TopicController.TOPICS_URI, organizationId)
        .then()
        .log()
        .all()
        .statusCode(httpStatus.value())
        .contentType(ContentType.JSON)
        .extract()
        .body()
        .as(response);
  }

  protected <T> T getTopic(
      UUID organizationId, UUID topicId, HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(EMPLOYEE_JWT_TOKEN)
        .get(TopicController.TOPIC_URI, organizationId, topicId)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected <T> T getTopicWithoutJwt(
      UUID organizationId, UUID topicId, HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .get(TopicController.TOPIC_URI, organizationId, topicId)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected <T> T updateTopic(
      UUID organizationId,
      UUID topicId,
      TopicUpdateRequest request,
      HttpStatus httpStatus,
      Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(EMPLOYEE_JWT_TOKEN)
        .body(request)
        .put(TopicController.TOPIC_URI, organizationId, topicId)
        .then()
        .log()
        .all()
        .statusCode(httpStatus.value())
        .contentType(ContentType.JSON)
        .extract()
        .body()
        .as(response);
  }

  protected TopicDetailResponse updateTopic(
      UUID organizationId, UUID topicId, TopicUpdateRequest request) {
    return updateTopic(organizationId, topicId, request, HttpStatus.OK, TopicDetailResponse.class);
  }

  protected <T> T updateTopicWithoutJwt(
      UUID organizationId,
      UUID topicId,
      TopicUpdateRequest request,
      HttpStatus httpStatus,
      Class<T> response) {
    return requestSpecification()
        .when()
        .body(request)
        .put(TopicController.TOPIC_URI, organizationId, topicId)
        .then()
        .log()
        .all()
        .statusCode(httpStatus.value())
        .contentType(ContentType.JSON)
        .extract()
        .body()
        .as(response);
  }

  protected <T> T listTopics(UUID organizationId, HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(EMPLOYEE_JWT_TOKEN)
        .get(TopicController.TOPICS_URI, organizationId)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected TopicListResponse listTopics(UUID organizationId) {
    return listTopics(organizationId, HttpStatus.OK, TopicListResponse.class);
  }

  protected <T> T listTopicsWithoutJwt(UUID organizationId, HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .get(TopicController.TOPICS_URI, organizationId)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected void deleteTopic(UUID organizationId, UUID topicId, HttpStatus status) {
    requestSpecification()
        .when()
        .auth()
        .oauth2(EMPLOYEE_JWT_TOKEN)
        .delete(TopicController.TOPIC_URI, organizationId, topicId)
        .then()
        .log()
        .all()
        .statusCode(status.value());
  }

  protected <T> T deleteTopicWithError(
      UUID organizationId, UUID topicId, HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .auth()
        .oauth2(EMPLOYEE_JWT_TOKEN)
        .delete(TopicController.TOPIC_URI, organizationId, topicId)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .extract()
        .as(response);
  }

  protected <T> T deleteTopicWithoutJwt(
      UUID organizationId, UUID topicId, HttpStatus status, Class<T> response) {
    return requestSpecification()
        .when()
        .delete(TopicController.TOPIC_URI, organizationId, topicId)
        .then()
        .log()
        .all()
        .statusCode(status.value())
        .contentType(ContentType.JSON)
        .extract()
        .as(response);
  }

  protected void checkFiltering(
      UUID organizationId,
      String filter,
      Integer expectedCount,
      TopicDetailResponse... expectedMessages) {

    final TopicListResponse response =
        requestSpecification()
            .when()
            .auth()
            .oauth2(EMPLOYEE_JWT_TOKEN)
            .get(TopicController.TOPICS_URI + filter, organizationId)
            .then()
            .statusCode(HttpStatus.OK.value())
            .extract()
            .body()
            .as(TopicListResponse.class);

    assertThat(response.totalCount()).isEqualTo(expectedCount);
    assertThat(response.items().size()).isEqualTo(expectedCount);

    final List<TopicDetailResponse> expectedMessageList = List.of(expectedMessages);
    response
        .items()
        .forEach(
            detail -> {
              final Optional<TopicDetailResponse> found =
                  expectedMessageList.stream()
                      .filter(r -> detail.getId().equals(r.getId()))
                      .findFirst();
              assertThat(found.isPresent()).isTrue();
            });
  }

  protected void checkSorting(
      UUID organizationId,
      String sort,
      Integer totalCount,
      TopicDetailResponse... expectedMessages) {

    final TopicListResponse response =
        requestSpecification()
            .when()
            .auth()
            .oauth2(EMPLOYEE_JWT_TOKEN)
            .get(TopicController.TOPICS_URI + "?sort=" + sort, organizationId)
            .then()
            .statusCode(HttpStatus.OK.value())
            .extract()
            .body()
            .as(TopicListResponse.class);

    assertThat(response.totalCount()).isEqualTo(totalCount);
    assertThat(response.items().size()).isEqualTo(expectedMessages.length);

    final List<TopicDetailResponse> expectedMessageList = List.of(expectedMessages);
    IntStream.range(0, response.items().size())
        .forEach(
            i ->
                assertThat(response.items().get(i).getId())
                    .isEqualTo(expectedMessageList.get(i).getId()));
  }
}
