'use client';

import React from 'react';
import axios from 'axios';
import { api, Topic, unwrap } from '@/utils';
import { Dialog, Input, Loader, PrimaryButton, SecondaryButton, TextField } from '@/lib/idsk';
import { useMutation } from '@tanstack/react-query';
import { useNotifications } from '@/hooks';

interface UpdateTopicDialogProps {
  topic: Topic | null;
  organizationId: string;
  onUpdate: () => void;
  onClose: () => void;
}

const UpdateTopicDialog: React.FC<UpdateTopicDialogProps> = ({ topic, organizationId, onUpdate, onClose }) => {
  const notify = useNotifications();

  const update = useMutation({
    mutationKey: ['topics'],
    mutationFn: (data: any) => unwrap(axios.put(`/api${api.topicsDetail(organizationId, data.id)}`, data)),
    onSuccess: () => {
      onUpdate();
    },
    onError: () => {
      onClose();
      notify({
        message: 'Tému sa nepodarilo upraviť.',
        variant: 'warning'
      });
    }
  });

  const updateTopic = (formData: FormData) => {
    const name = formData.get('name');
    const description = formData.get('description');
    update.mutate({
      ...topic,
      name,
      description
    });
  };

  const isLoading = update.status === 'pending';

  return (
    !!topic && (
      <form action={updateTopic}>
        <Dialog
          opened={!!topic}
          toggleOpened={onClose}
          title="Upraviť tému"
          primaryButton={<PrimaryButton disabled={isLoading}>Upraviť</PrimaryButton>}
          secondaryButton={<SecondaryButton onClick={onClose}>Zatvoriť</SecondaryButton>}
        >
          <div className="w-full tb2:min-w-[42rem] flex flex-col justify-start items-start relative">
            <Input label="Názov" name="name" defaultValue={topic?.name || ''} required mandatory fullWidth />
            <TextField
              label="Popis"
              name="description"
              defaultValue={topic?.description || ''}
              required
              mandatory
              fullWidth
            />

            {isLoading && (
              <div className="bg-white flex justify-center items-center absolute top-0 left-0 w-full h-full">
                <Loader />
              </div>
            )}
          </div>
        </Dialog>
      </form>
    )
  );
};

export default UpdateTopicDialog;
