'use client';

import React, { useRef, useState } from 'react';
import { ContentLayout, MessageList, NewMessageDialog, withPermission } from '@/components';
import { Pagination, SearchBar } from '@/lib/idsk';
import { useRouter, useSearchParams } from 'next/navigation';

const PAGE_SIZE = 8;

const Mailbox: React.FC = () => {
  const [totalCount, setTotalCount] = useState<number>(0);
  const router = useRouter();
  const params = useSearchParams();
  const page = Number(params.get('page') || 1);
  const search = params.get('search') || '';

  const ref = useRef<HTMLInputElement>(null);

  const onPageChange = (newPage: number) => {
    const newParams = new URLSearchParams(params.toString());
    newParams.set('page', newPage.toString());
    router.push(`/spravy?${newParams.toString()}`);
  };

  const onSearch = () => {
    const newParams = new URLSearchParams(params.toString());
    newParams.set('search', ref.current?.value || '');
    newParams.set('page', '1');
    router.push(`/spravy?${newParams.toString()}`);
  };

  const onClearSearch = () => {
    const newParams = new URLSearchParams(params.toString());
    newParams.delete('search');
    newParams.set('page', '1');
    router.push(`/spravy?${newParams.toString()}`);
  };

  return (
    <ContentLayout
      title="Správy"
      actions={
        <>
          <SearchBar
            ref={ref}
            searchbarSize="medium"
            showCancelButton={!!search}
            buttonOnClick={onSearch}
            onCancel={onClearSearch}
            fullWidth
          />
          <NewMessageDialog />
        </>
      }
    >
      <MessageList page={page} pageSize={PAGE_SIZE} search={search} onLoad={(data) => setTotalCount(data.totalCount)} />
      {totalCount > PAGE_SIZE && (
        <div className="mt-7 w-full flex justify-end">
          <Pagination
            forcePage={page}
            pageCount={Math.ceil(totalCount / PAGE_SIZE)}
            caption={''}
            onPageChange={({ selected }) => onPageChange(selected)}
          />
        </div>
      )}
    </ContentLayout>
  );
};

export default withPermission(Mailbox, 'ACTION_MANAGE_MESSAGE');
