'use client';

import React, { useMemo, useState } from 'react';
import { cn, Fault, RequestError } from '@/utils';
import { ErrorImg, ArrowDropDownIcon, InformationBanner, BaseButton } from '@/lib/idsk';

interface ErrorProps {
  err: RequestError | null;
  transparent?: boolean;
  asBanner?: boolean;
}

const FaultItem: React.FC<{ fault: Fault }> = ({ fault }) => {
  const message = useMemo(() => {
    const listOfParams = (fault as any).faultMessageFormat || (fault as any).faultMessageParams || [];
    if (listOfParams.length > 0) {
      return listOfParams.reduce((result: string, param: 'string') => result.replace('%s', param), fault.faultMessage);
    }

    return fault.faultMessage;
  }, [fault]);

  return (
    <div className="flex flex-col gap-1.5">
      <div>
        <b className="block">ServiceId:</b>
        <span>{fault.serviceId || '-'}</span>
      </div>
      <div>
        <b className="block">Code:</b>
        <span>{fault.faultCode || '-'}</span>
      </div>
      <div>
        <b className="block">Message:</b>
        <span>{message || '-'}</span>
      </div>
    </div>
  );
};

const ErrorWrapper: React.FC<{
  children: React.ReactNode;
  asBanner?: boolean;
  err: RequestError;
  transparent?: boolean;
}> = ({ children, err, transparent = false, asBanner = false }) => {
  if (asBanner) {
    return (
      <InformationBanner
        className="custom-banner"
        title={`${err.status} - ${err.statusText}`}
        variant="warning"
        hideCloseButton
      >
        {children}
      </InformationBanner>
    );
  }

  return (
    <div
      className={cn('idsk-text-body flex flex-col items-center justify-center p-8', {
        'bg-white w-full border border-neutral-300 rounded-lg': !transparent
      })}
    >
      {children}
    </div>
  );
};

const Error: React.FC<ErrorProps> = ({ err, asBanner = false, transparent = false }) => {
  const [isOpen, setIsOpen] = useState<boolean>(false);

  if (!err) return null;

  return (
    <ErrorWrapper err={err} transparent={transparent} asBanner={asBanner}>
      {!asBanner && (
        <>
          <ErrorImg className="max-w-32" />
          <h3>{`${err.status} - ${err.statusText}`}</h3>
        </>
      )}
      <p>{err.message}</p>

      <BaseButton
        className={cn('flex items-center hover:underline text-primary-medium', { 'mx-auto mt-3': !asBanner })}
        onClick={() => setIsOpen((p) => !p)}
      >
        <span>Detail</span>
        <ArrowDropDownIcon style={{ rotate: isOpen ? '180deg' : '0deg' }} />
      </BaseButton>
      {isOpen && (
        <div
          className={cn('w-full rounded-md mt-4 text-left text-base flex flex-col gap-1.5', {
            'p-2.5 bg-neutral-200 ': !asBanner
          })}
        >
          <div>
            <b className="block">CorrelationId:</b>
            <span>{err.correlationId || '-'}</span>
          </div>
          {err.fault.map((fault, index) => (
            <FaultItem key={index} fault={fault} />
          ))}
        </div>
      )}
    </ErrorWrapper>
  );
};

export default Error;
