package sk.kosice.konto.kkmessageservice.idp;

import static sk.kosice.konto.kkmessageservice.domain.organization.error.OrganizationErrorCode.ORGANIZATION_DOES_NOT_EXIST;
import static sk.kosice.konto.kkmessageservice.domain.permission.error.PermissionErrorCode.USER_DOES_NOT_EXIST;
import static sk.kosice.konto.kkmessageservice.idp.error.AzureIdpErrorCode.AZURE_IDP_EXTERNAL_ERROR;

import com.microsoft.graph.directoryobjects.item.getmembergroups.GetMemberGroupsPostRequestBody;
import com.microsoft.graph.models.Group;
import com.microsoft.graph.models.odataerrors.ODataError;
import com.microsoft.graph.serviceclient.GraphServiceClient;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import sk.kosice.konto.kkmessageservice.business.organization.port.outbound.QueryOrganizationPort;
import sk.kosice.konto.kkmessageservice.domain.common.error.BusinessException;
import sk.kosice.konto.kkmessageservice.domain.organization.entity.OrganizationEntity;
import sk.kosice.konto.kkmessageservice.idp.mapper.AzureIdpMapper;

@Component
public class AzureIdpRepository implements QueryOrganizationPort {

  private static final Logger log = LoggerFactory.getLogger(AzureIdpRepository.class);
  private final GraphServiceClient graphClient;

  @Autowired
  public AzureIdpRepository(GraphServiceClient graphClient) {
    this.graphClient = graphClient;
  }

  @Override
  public List<OrganizationEntity> list(Optional<UUID> rootGroupId) throws BusinessException {
    log.info("Retrieving groups list from Azure with root group id: {}", rootGroupId);

    try {
      if (rootGroupId.isPresent()) {
        return Optional.ofNullable(
                graphClient.groups().byGroupId(rootGroupId.get().toString()).get().getMembers())
            .orElse(Collections.emptyList())
            .stream()
            .map(
                g -> {
                  final var group = graphClient.groups().byGroupId(g.getId()).get();

                  return AzureIdpMapper.map(group);
                })
            .collect(Collectors.toList());
      } else {
        return AzureIdpMapper.map(
            graphClient
                .groups()
                .get(r -> r.queryParameters.filter = "startswith(displayName, 'kk-')")
                .getValue());
      }
    } catch (Exception e) {
      log.error("Error occurred while retrieving groups from Azure: {}", e.getMessage(), e);
      throw AZURE_IDP_EXTERNAL_ERROR.createError(e.getMessage()).convertToException();
    }
  }

  @Override
  public OrganizationEntity findOne(UUID organizationId) throws BusinessException {
    log.info("Retrieving group detail from Azure with group id: {}", organizationId);
    Group group = null;
    try {
      group = graphClient.groups().byGroupId(organizationId.toString()).get();
    } catch (Exception e) {
      log.error("Error occurred while retrieving group from Azure: {}", e.getMessage(), e);
      throw AZURE_IDP_EXTERNAL_ERROR.createError(e.getMessage()).convertToException();
    }

    if (group == null) {
      throw ORGANIZATION_DOES_NOT_EXIST.createError(organizationId).convertToException();
    }
    return AzureIdpMapper.map(group);
  }

  @Override
  public List<OrganizationEntity> getUserGroups(String userId) {
    try {
      return AzureIdpMapper.map(
          graphClient.users().byUserId(userId).memberOf().graphGroup().get().getValue());
    } catch (ODataError e) {
      if (e.getResponseStatusCode() == 404) {
        log.debug("User with ID '{}' not found in Azure Id", e.getMessage(), e);
        throw USER_DOES_NOT_EXIST.createError(userId).convertToException();
      } else {
        log.error("Error occurred while retrieving group from Azure: {}", e.getMessage(), e);
        throw AZURE_IDP_EXTERNAL_ERROR.createError(e.getMessage()).convertToException();
      }
    }
  }

  @Override
  public List<UUID> getUserGroupIds(String userId) {
    try {
      var requestGetMemberGroups = new GetMemberGroupsPostRequestBody();
      requestGetMemberGroups.setSecurityEnabledOnly(true);
      return graphClient
          .directoryObjects()
          .byDirectoryObjectId(userId)
          .getMemberGroups()
          .post(requestGetMemberGroups)
          .getValue()
          .parallelStream()
          .map(UUID::fromString)
          .collect(Collectors.toList());
    } catch (ODataError e) {
      if (e.getResponseStatusCode() == 404) {
        log.debug("User with ID '{}' not found in Azure Id", e.getMessage(), e);
        throw USER_DOES_NOT_EXIST.createError(userId).convertToException();
      } else {
        log.error("Error occurred while retrieving group from Azure: {}", e.getMessage(), e);
        throw AZURE_IDP_EXTERNAL_ERROR.createError(e.getMessage()).convertToException();
      }
    }
  }
}
