package sk.kosice.konto.kkmessageservice.repository;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sk.kosice.konto.kkmessageservice.domain.common.error.BusinessException;
import sk.kosice.konto.kkmessageservice.repository.error.ConstraintErrorCode;
import sk.kosice.konto.kkmessageservice.repository.error.DbErrorCode;
import sk.kosice.konto.kkmessageservice.repository.error.RepositoryErrorHandler;

public abstract class JooqRepository {

  protected final DSLContext dslContext;

  protected static final Logger log = LoggerFactory.getLogger(JooqRepository.class);

  protected JooqRepository(DSLContext dslContext) {
    this.dslContext = dslContext;
  }

  protected static void handleBatchUpdateResults(
      Supplier<int[]> supplier, List<?> dataObject, ConstraintErrorCode... errorCodes)
      throws BusinessException {
    try {
      final int numOfInserts = Arrays.stream(supplier.get()).sum();

      if (numOfInserts < 1) {
        dbError("No record has been updated.");
      }

    } catch (Exception e) {
      throw (BusinessException) RepositoryErrorHandler.handle(e, dataObject, errorCodes);
    }
  }

  protected static void handleDeleteResult(
      Supplier<Integer> supplier, Object dataObject, ConstraintErrorCode... errorCodes)
      throws BusinessException {
    try {
      final Integer numOfDeletes = supplier.get();

      if (numOfDeletes.equals(1)) {
        // empty
      } else if (numOfDeletes.equals(0)) {
        dbError("No record has been deleted.");
      } else {
        dbError("Multiple records have been deleted.");
      }
    } catch (Exception e) {
      throw (BusinessException) RepositoryErrorHandler.handle(e, dataObject, errorCodes);
    }
  }

  protected static void handleInsertResult(
      Supplier<Integer> supplier, Object dataObject, ConstraintErrorCode... errorCodes)
      throws BusinessException {
    try {
      final Integer numOfInserts = supplier.get();

      if (numOfInserts.equals(0)) {
        dbError("No record has been inserted.");
      } else if (numOfInserts.compareTo(1) > 0) {
        dbError("Multiple records have been inserted.");
      }
    } catch (Exception e) {
      throw (BusinessException) RepositoryErrorHandler.handle(e, dataObject, errorCodes);
    }
  }

  protected static void handleUpdateResult(
      Supplier<Integer> supplier, Object dataObject, ConstraintErrorCode... errorCodes)
      throws BusinessException {
    try {
      final Integer numOfUpdates = supplier.get();

      if (numOfUpdates.equals(1)) {
        // empty
      } else if (numOfUpdates.equals(0)) {
        dbError("No record has been updated.");
      } else {
        dbError("Multiple records have been updated.");
      }
    } catch (Exception e) {
      throw (BusinessException) RepositoryErrorHandler.handle(e, dataObject, errorCodes);
    }
  }

  protected <T> T handleFindOneResult(
      Supplier<List<T>> supplier, Object dataObject, ConstraintErrorCode... errorCodes)
      throws BusinessException {
    try {
      final List<T> results = supplier.get();

      if (results.isEmpty()) {
        createEntityDoesNotExistError(dataObject);
      } else if (results.size() == 1) {
        return results.get(0);
      } else {
        dbError("Too many records were found.");
      }
      return null;
    } catch (Exception e) {
      throw (BusinessException) RepositoryErrorHandler.handle(e, dataObject, errorCodes);
    }
  }

  protected static <T> List<T> handleListedResult(Supplier<List<T>> supplier) {
    try {
      return supplier.get();
    } catch (Exception e) {
      dbError("Records can not be listed due to database error.");
    }
    return Collections.emptyList();
  }

  protected static void dbError(String message) {
    throw DbErrorCode.DB_ERROR.createError(message).convertToException();
  }

  protected abstract void createEntityDoesNotExistError(Object id);
}
