package sk.kosice.konto.kkmessageservice.repository.topic;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import org.junit.jupiter.api.Test;
import sk.kosice.konto.kkmessageservice.domain.common.listing.ImmutableListOrdering;
import sk.kosice.konto.kkmessageservice.domain.common.listing.ListOrdering;
import sk.kosice.konto.kkmessageservice.domain.common.listing.PageSize;
import sk.kosice.konto.kkmessageservice.domain.common.listing.TopicListingAttribute;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.ListOfTopics;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.TopicEntity;
import sk.kosice.konto.kkmessageservice.domain.topic.query.ImmutableTopicListingQuery;

public class JooqTopicRepositoryListTest extends JooqTopicRepositoryTest {

  @Test
  public void thatEmptyTopicListCanBeRetrieved() {
    final ListOfTopics messages = listTopics(topicRepository, UUID.randomUUID());
    assertThat(messages.items()).isEmpty();
    assertThat(messages.totalCount()).isZero();
  }

  @Test
  public void thatTopicListCanBeRetrieved() {
    final var organizationId = UUID.randomUUID();
    createTopicsForFilteringAndSorting(organizationId);
    final ListOfTopics topics = listTopics(topicRepository, organizationId);

    assertThat(topics.items()).isNotEmpty();
    assertThat(topics.totalCount()).isEqualTo(3);
  }

  @Test
  public void thatTopicListCanBeFilteredByName() {
    final var organizationId = UUID.randomUUID();

    final List<TopicEntity> messages = createTopicsForFilteringAndSorting(organizationId);
    findWithFilter(organizationId, "name==\"" + messages.get(0).name() + "\"", 1);
  }

  @Test
  public void thatTopicListCanBeFilteredByDescription() {
    final var organizationId = UUID.randomUUID();

    final List<TopicEntity> messages = createTopicsForFilteringAndSorting(organizationId);
    findWithFilter(organizationId, "description==\"" + messages.get(0).description() + "\"", 1);
  }

  @Test
  public void thatTopicListCanBeSortedByName() {
    final var organizationId = UUID.randomUUID();
    final List<TopicEntity> topics = createTopicsForFilteringAndSorting(organizationId);

    findSorted(
        organizationId,
        new TopicEntity[] {topics.get(0), topics.get(1), topics.get(2)},
        ImmutableListOrdering.of(TopicListingAttribute.NAME, true));

    findSorted(
        organizationId,
        new TopicEntity[] {topics.get(2), topics.get(1), topics.get(0)},
        ImmutableListOrdering.of(TopicListingAttribute.NAME, false));
  }

  @Test
  public void thatTopicListCanBePaginated() {
    final UUID organizationId = UUID.randomUUID();
    final List<TopicEntity> topics = createTopicsForFilteringAndSorting(organizationId);

    findPage(
        organizationId,
        1,
        1,
        ImmutableListOrdering.of(TopicListingAttribute.NAME, true),
        topics.get(0));

    findPage(
        organizationId,
        2,
        2,
        ImmutableListOrdering.of(TopicListingAttribute.NAME, true),
        topics.get(2));

    findPage(
        organizationId,
        3,
        1,
        ImmutableListOrdering.of(TopicListingAttribute.NAME, false),
        topics.get(2),
        topics.get(1),
        topics.get(0));
  }

  private List<TopicEntity> createTopicsForFilteringAndSorting(UUID organizationId) {

    final TopicEntity t1 =
        persistAndFindTopic(
            topicRepository,
            prepareTopicData()
                .withName("name1")
                .withDescription("description1")
                .withOrganizationId(organizationId));

    final TopicEntity t2 =
        persistAndFindTopic(
            topicRepository,
            prepareTopicData()
                .withName("name2")
                .withDescription("description2")
                .withOrganizationId(organizationId));

    final TopicEntity t3 =
        persistAndFindTopic(
            topicRepository,
            prepareTopicData()
                .withName("name3")
                .withDescription("description3")
                .withOrganizationId(organizationId));

    return List.of(t1, t2, t3);
  }

  private void findWithFilter(UUID organizationId, String rsqlQuery, int expectedNumOfResults) {
    final ListOfTopics listOfTopics =
        topicRepository.list(
            ImmutableTopicListingQuery.builder()
                .pageSize(PageSize.of(100, 100))
                .organizationId(organizationId)
                .addAllOrderings(Collections.emptyList())
                .rsqlQuery(Optional.of(rsqlParser.parse(rsqlQuery)))
                .build());

    assertThat(listOfTopics.totalCount()).isEqualTo(expectedNumOfResults);
  }

  @SafeVarargs
  private void findSorted(
      UUID organizationId,
      TopicEntity[] expectedOrder,
      ListOrdering<TopicListingAttribute>... orderings) {
    final ListOfTopics listOfTopics =
        topicRepository.list(
            ImmutableTopicListingQuery.builder()
                .pageSize(PageSize.of(100, 100))
                .organizationId(organizationId)
                .addAllOrderings(List.of(orderings))
                .rsqlQuery(Optional.empty())
                .build());

    assertThat(listOfTopics.items()).containsExactly(expectedOrder);
  }

  private void findPage(
      UUID organizationId,
      int pageSize,
      int pageNumber,
      ListOrdering<TopicListingAttribute> ordering,
      TopicEntity... expectedOrder) {
    final ListOfTopics listOfTopics =
        topicRepository.list(
            ImmutableTopicListingQuery.builder()
                .pageSize(PageSize.of(100, pageSize))
                .organizationId(organizationId)
                .rsqlQuery(Optional.empty())
                .page(pageNumber)
                .addAllOrderings(List.of(ordering))
                .build());

    assertThat(listOfTopics.items()).containsExactly(expectedOrder);
  }
}
