package sk.kosice.konto.kkmessageservice.restapi.controller;

import static sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingRequestParams.PARAM_FILTER;
import static sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingRequestParams.PARAM_PAGE;
import static sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingRequestParams.PARAM_PAGE_SIZE;
import static sk.kosice.konto.kkmessageservice.domain.common.listing.common.ListingRequestParams.PARAM_SORT;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;
import java.util.Optional;
import java.util.UUID;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import sk.kosice.konto.kkmessageservice.business.organization.port.inbound.FindSubscribedOrganizationsByRecipientUseCase;
import sk.kosice.konto.kkmessageservice.business.organization.port.inbound.ListOrganizationUseCase;
import sk.kosice.konto.kkmessageservice.domain.common.listing.SubscriptionListingAttribute;
import sk.kosice.konto.kkmessageservice.restapi.command.OrganizationQueryFactory;
import sk.kosice.konto.kkmessageservice.restapi.common.enumeration.Platform;
import sk.kosice.konto.kkmessageservice.restapi.dto.organization.OrganizationListResponse;
import sk.kosice.konto.kkmessageservice.restapi.mapper.OrganizationResponseMapper;
import sk.kosice.konto.kkmessageservice.restapi.springdoc.restapi.springdoc.annotation.FilterParameter;
import sk.kosice.konto.kkmessageservice.restapi.springdoc.restapi.springdoc.annotation.SortParameter;

@Tag(name = "Organization", description = "Organization management")
@RestController
public class OrganizationController extends BaseController {
  public static final String ORGANIZATIONS_URI = BASE_V1_URI + "/organizations";
  public static final String ORGANIZATIONS_BY_KID_URI =
      BASE_V1_URI + "/recipients/{kid}/organizations";

  private final ListOrganizationUseCase listOrganizationUseCase;
  private final FindSubscribedOrganizationsByRecipientUseCase
      findSubscribedOrganizationsByRecipientUseCase;

  @Autowired
  public OrganizationController(
      ListOrganizationUseCase listOrganizationUseCase,
      FindSubscribedOrganizationsByRecipientUseCase findSubscribedOrganizationsByRecipientUseCase) {
    this.listOrganizationUseCase = listOrganizationUseCase;
    this.findSubscribedOrganizationsByRecipientUseCase =
        findSubscribedOrganizationsByRecipientUseCase;
  }

  @Operation(
      summary = "List of organizations.",
      description = "Returns a list of messages matching given conditions.")
  @ApiResponses({
    @ApiResponse(
        responseCode = "200",
        description = "Successful response with organizations matching given conditions.",
        content = @Content(schema = @Schema(implementation = OrganizationListResponse.class)))
  })
  @GetMapping(value = ORGANIZATIONS_URI, produces = MediaType.APPLICATION_JSON_VALUE)
  public ResponseEntity<?> list(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @FilterParameter(
              listingAttributeClass = SubscriptionListingAttribute.class,
              example = "body=like=*Pozvánka*")
          @RequestParam(name = PARAM_FILTER, defaultValue = "")
          String filter,
      @Parameter(description = "Page query parameter.")
          @RequestParam(name = PARAM_PAGE, defaultValue = "1")
          Integer page,
      @Parameter(description = "Page size query parameter.")
          @RequestParam(name = PARAM_PAGE_SIZE, defaultValue = "10")
          Integer pageSize,
      @SortParameter(listingAttributeClass = SubscriptionListingAttribute.class)
          @RequestParam(name = PARAM_SORT, defaultValue = "")
          String orderings) {
    log.info("Calling OrganizationController GET list: [{}]", ORGANIZATIONS_URI);
    return ResponseEntity.ok(
        OrganizationResponseMapper.map(
            listOrganizationUseCase.execute(OrganizationQueryFactory.map(Optional.empty()))));
  }

  @Operation(
      summary = "List of organizations for kid.",
      description = "Returns a list of messages matching given conditions.")
  @ApiResponses({
    @ApiResponse(
        responseCode = "200",
        description = "Successful response with organizations matching given conditions.",
        content = @Content(schema = @Schema(implementation = OrganizationListResponse.class)))
  })
  @GetMapping(value = ORGANIZATIONS_BY_KID_URI, produces = MediaType.APPLICATION_JSON_VALUE)
  public ResponseEntity<?> listByKid(
      @AuthenticationPrincipal @Parameter(hidden = true) Jwt jwt,
      @Parameter(
              description = "rest.api.common.header.correlationId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_CORRELATION_ID)
          UUID correlationId,
      @Parameter(
              description = "rest.api.common.header.appId.description",
              required = true,
              example = "6ba7b810-9dad-11d1-80b4-00c04fd430c6")
          @RequestHeader(name = HEADER_X_APP_ID)
          UUID appId,
      @Parameter(
              description = "rest.api.common.header.appVersion.description",
              required = true,
              example = "1.0.0")
          @RequestHeader(name = HEADER_X_APP_VERSION)
          String appVersion,
      @Parameter(
              description = "rest.api.common.header.appPlatform.description",
              required = true,
              example = "WEB")
          @RequestHeader(name = HEADER_X_PLATFORM)
          Platform appPlatform,
      @Parameter(
              description = "Unique identifier of recipient.",
              required = true,
              example = "a3fe6b7f-4a69-4114-ab50-ad4153a3afd1")
          @PathVariable(PARAM_KID)
          UUID kid) {
    log.info("Calling OrganizationController GET list: [{}]", ORGANIZATIONS_BY_KID_URI);
    return ResponseEntity.ok(
        OrganizationResponseMapper.map(
            findSubscribedOrganizationsByRecipientUseCase.execute(
                OrganizationQueryFactory.map(kid))));
  }
}
