package sk.kosice.konto.kkmessageservice.business.permission.service;

import jakarta.inject.Inject;
import jakarta.inject.Named;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import sk.kosice.konto.kkmessageservice.business.AbstractService;
import sk.kosice.konto.kkmessageservice.business.businessconfig.BusinessConfigProvider;
import sk.kosice.konto.kkmessageservice.business.organization.port.outbound.QueryOrganizationPort;
import sk.kosice.konto.kkmessageservice.business.permission.port.inbound.ListPermissionUseCase;
import sk.kosice.konto.kkmessageservice.domain.common.enumeration.Permission;
import sk.kosice.konto.kkmessageservice.domain.common.enumeration.Role;
import sk.kosice.konto.kkmessageservice.domain.permission.entity.ListOfPermissions;
import sk.kosice.konto.kkmessageservice.domain.permission.query.PermissionListingQuery;

@Named
public class ListPermissionService extends AbstractService implements ListPermissionUseCase {

  private final QueryOrganizationPort queryOrganizationPort;
  private final BusinessConfigProvider businessConfigProvider;

  @Inject
  public ListPermissionService(
      QueryOrganizationPort queryOrganizationPort, BusinessConfigProvider businessConfigProvider) {
    this.queryOrganizationPort = queryOrganizationPort;
    this.businessConfigProvider = businessConfigProvider;
  }

  @Override
  public ListOfPermissions execute(PermissionListingQuery query) {
    final var userOrganization = queryOrganizationPort.getUserGroups(query.userId().toString());
    final Set<Permission> permissions =
        new HashSet<>() {
          {
            if (userOrganization.stream()
                .anyMatch(
                    o ->
                        businessConfigProvider
                            .getBusinessConfig()
                            .adminRoleId()
                            .equals(o.id().toString()))) {
              addAll(getPermissionsByRole(Role.ADMIN));
            }
            if (userOrganization.stream()
                .anyMatch(
                    o ->
                        businessConfigProvider
                            .getBusinessConfig()
                            .adminGlobalRoleId()
                            .equals(o.id().toString()))) {
              addAll(getPermissionsByRole(Role.ADMIN_GLOBAL));
            }
          }
        };

    final var mappedPermissions =
        userOrganization.stream()
            .filter(
                group ->
                    !businessConfigProvider
                            .getBusinessConfig()
                            .adminGlobalRoleId()
                            .equals(group.id().toString())
                        && !businessConfigProvider
                            .getBusinessConfig()
                            .adminRoleId()
                            .equals(group.id().toString()))
            .map(group -> Map.of(group, permissions))
            .flatMap(e -> e.entrySet().stream())
            .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));

    return new ListOfPermissions(query.userId(), mappedPermissions);
  }

  private List<Permission> getPermissionsByRole(Role role) {
    return Arrays.stream(Permission.values()).filter(p -> p.getRoles().contains(role)).toList();
  }
}
