package sk.kosice.konto.kknotificationservice.business.email.port.service;

import jakarta.inject.Inject;
import jakarta.inject.Named;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sk.kosice.konto.kkmessageservice.domain.common.enumeration.BodyType;
import sk.kosice.konto.kknotificationservice.business.AbstractService;
import sk.kosice.konto.kknotificationservice.business.businessconfig.BusinessConfigProvider;
import sk.kosice.konto.kknotificationservice.business.email.port.inbound.CreateEmailFromMessageUseCase;
import sk.kosice.konto.kknotificationservice.business.recipient.port.outbound.QueryRecipientPort;
import sk.kosice.konto.kknotificationservice.domain.email.command.ImmutableSendEmailCommand;
import sk.kosice.konto.kknotificationservice.domain.email.command.SendEmailCommand;
import sk.kosice.konto.kknotificationservice.domain.email.entity.ImmutableSenderInfo;
import sk.kosice.konto.kknotificationservice.domain.message.entity.MessageEntity;
import sk.kosice.konto.kknotificationservice.domain.recipient.entity.RecipientEntity;
import sk.kosice.konto.kknotificationservice.domain.shared.enumeration.EmailBodyType;

@Named
public class CreateEmailFromMessageService extends AbstractService
    implements CreateEmailFromMessageUseCase {
  protected static final Logger log = LoggerFactory.getLogger(CreateEmailFromMessageService.class);
  private static final String EMAIL_PATTERN =
      "^(?![!#$%&'*+/=?^_`{|}~])[a-zA-Z0-9!#$%&'*+/=?^_`{|}~-]+"
          + "(?:\\.[a-zA-Z0-9!#$%&'*+/=?^_`{|}~-]+)*"
          + "@(?:(?:[a-zA-Z0-9](?:[a-zA-Z0-9-]*[a-zA-Z0-9])?\\.)+"
          + "[a-zA-Z0-9](?:[a-zA-Z0-9-]*[a-zA-Z0-9])?|\\[(?:(?:25[0-5]|2[0-4][0-9]|"
          + "[01]?[0-9][0-9]?)\\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|"
          + "[a-zA-Z0-9-]*[a-zA-Z0-9]:(?:[\\x01-\\x08\\x0b\\x0c\\x0e-\\x1f\\x21-\\x5a\\x53-\\x7f]|"
          + "\\\\[\\x01-\\x09\\x0b\\x0c\\x0e-\\x7f])+)\\])$";

  private final BusinessConfigProvider businessConfigProvider;
  private final QueryRecipientPort queryRecipientPort;

  @Inject
  public CreateEmailFromMessageService(
      BusinessConfigProvider businessConfigProvider, QueryRecipientPort queryRecipientPort) {
    this.businessConfigProvider = businessConfigProvider;
    this.queryRecipientPort = queryRecipientPort;
  }

  @Override
  public SendEmailCommand execute(MessageEntity entity) {
    List<String> recipients = new ArrayList<>();

    if (entity.recipientKid().isEmpty()) {
      recipients.addAll(
          queryRecipientPort.listByTopic(entity.topicId()).stream()
              .map(RecipientEntity::email)
              .filter(CreateEmailFromMessageService::validateEmail)
              .toList());
    } else {
      recipients.add(queryRecipientPort.findOne(entity.recipientKid().get()).email());
    }

    return ImmutableSendEmailCommand.builder()
        .messageTitle(entity.title())
        .messageBody(entity.body())
        .messageBodyType(resolveEmailBodyType(entity.bodyType()))
        .senderInfo(
            ImmutableSenderInfo.builder()
                .email(businessConfigProvider.getBusinessConfig().senderEmail())
                .name(businessConfigProvider.getBusinessConfig().senderName())
                .build())
        .addAllRecipients(recipients)
        .build();
  }

  private EmailBodyType resolveEmailBodyType(BodyType bodyType) {
    return switch (bodyType) {
      case TEXT, FORMATTED -> EmailBodyType.TEXT;
      case HTML -> EmailBodyType.HTML;
    };
  }

  public static boolean validateEmail(String email) {
    Pattern pattern = Pattern.compile(EMAIL_PATTERN);
    Matcher matcher = pattern.matcher(email);
    return matcher.matches();
  }
}
