/**
 * FloatingPlayer component.
 *
 * @author: Vaclav Hradec <vaclav@mihybrid.com>
 *
 */

'use strict';

import { addClass, removeClass, hasClass } from "mihbbtv/lib/util/classes";
import BaseComponent from "mihbbtv/lib/components/base";
import KeyHandler from "mihbbtv/lib/hbbtv/keyHandler";
import Player from "./player";
import Config from "../config";

class FloatingPlayer extends BaseComponent {

  static getInstance(cfg, parent) {
    if (! FloatingPlayer.instance) {
      FloatingPlayer.instance = new FloatingPlayer(cfg, parent);
    }

    return FloatingPlayer.instance;
  }

  constructor(cfg, parent, className = 'floatingPlayer loading') {
    super(cfg.container, parent, className);

    this.init();
  }

  init() {

    if (this.container && ! this.elm) {
      this.elm = document.getElementById(this.container);
    }

    this.elm.innerHTML = this.markup();
    this.playingElm = null;

    this.player = new Player({
      container: 'appPlayer',
      loop: false,
      GUI: 'horizontal',
      mediaStyle: {
        width: 'auto',
        height: 'auto',
        top: '0',
        left:'0'
      },
      type: 'audio',
      autoplay: true,
      exitPlayerCb: () => {
        this.stopped();
      },
      startPlayerCb: () => {
        this.started();
      },
    }, this.elm);
  }

  markup() {
    return '<div class="buttons">' +
      '<div class="btn prev"></div>' +
      '<div id="' + this.container + 'PlayPauseButton" class="btn pause"></div>' +
      '<div class="btn next"></div>' +
    '</div>' +
    '<div id="' + this.container + 'Track" class="track"></div>' +
    '<div class="status" id="navigation">' +
      '<div class="elapsed" id="current-time">00:00</div>' +
      '<div class="progress"><div class="progressBar" id="passed-navigation"></div></div>' +
      '<div class="remain" id="remaining-time">--:--</div>' +
    '</div>'
    ;
  }

  stopped() {
    // console.log(`Floating player is stopped.`);
  }
  started() {
    // console.log(`Floating player is started.`);

    removeClass(this.elm, 'loading');
  }

  playPause() {
    this.player.pausePlay();

    const btn = document.getElementById(this.container + 'PlayPauseButton');
    if (btn) {
      if (this.player.isPlayling()) {
        removeClass(btn, 'play');
        addClass(btn, 'pause');
      } else {
        removeClass(btn, 'pause');
        addClass(btn, 'play');
      }
    }
  }

  play(item, elm) {
    removeClass(this.elm, 'audio');
    removeClass(this.elm, 'live');

    if (item) {
      addClass(this.elm, item.type);
    }

    const btn = document.getElementById(this.container + 'PlayPauseButton');
    if (! btn) {
      return false;
    }

    if (! this.player) {
      return false;
    }

    // console.log(item, elm);

    // underlying player handling
    if ((this.player.currentMedia() === item.audio) && (this.player.isPlayling() || this.player.isPaused())) {
      // console.log(`floating Player: pause/play`);

      this.player.pausePlay();
    } else {
      // console.log(`floating Player: load and play`, item);
      addClass(this.elm, 'loading');

      this.player.load({
        playlist: [{
            urls: item.audio,
            opt: {
              headline: item.headline,
              dur: item.duration,
              id: item.id,
              show: (item.show) ? (item.show.id + ': ' + item.show.title) : null,
              creator: item.channel,
              type: item.type,
            }
        }]
      });
    }

    if (hasClass(elm, 'playing')) {

      removeClass(elm, 'playing');

      removeClass(btn, 'pause');
      addClass(btn, 'play');
    } else {

      removeClass(btn, 'play');
      addClass(btn, 'pause');

      if (this.playingElm) {
        removeClass(this.playingElm, 'playing');
      }

      this.playingElm = elm;
      if (this.playingElm) {
        addClass(this.playingElm, 'playing');
      }
    }

    const trackElm = document.getElementById(this.container + 'Track');

    if (! trackElm) {
      return false;
    }

    // console.log(item);

    trackElm.innerText = item.headline || item.title;

    this.show();
  }

  stop() {
    this.player.stop();

    if (this.playingElm) {
      removeClass(this.playingElm, 'playing');
    }

    this.hide();
  }

  keyHandlerButtons(kc) {
    // console.log(`FloatingPlayer`, kc);

    if ((kc === KeyHandler.keycodes.VK_PLAY) || (kc === KeyHandler.keycodes.VK_PLAY_PAUSE) || (kc === KeyHandler.keycodes.VK_PAUSE)) {
      // console.log(`Play/pause`);

      if (this.player.isPlayling() || this.player.isPaused()) {
        this.playPause();

        return true;
      }
    }

    if (kc === KeyHandler.keycodes.VK_STOP) {
      // console.log(`Stop`);

      this.stop();

      return true;
    }

    if (kc === KeyHandler.keycodes.VK_FAST_FWD) {
      // console.log(`forward`);
      this.player.forward();
      return true;
    }

    if (kc === KeyHandler.keycodes.VK_REWIND) {
      // console.log(`rewind`);
      this.player.backward();
      return true;
    }
  }


}

FloatingPlayer.instance = null;

export default FloatingPlayer;
