import React from 'react';
import { useTranslations } from 'next-intl';
import { Pagination as IdskPagination, PaginationProps as IdskPaginationProps } from '@/lib/idsk';

interface PaginationProps extends Omit<Partial<IdskPaginationProps>, 'onPageChange' | 'caption'> {
  page: number;
  pageSize: number;
  totalCount: number;
  caption?: boolean | string;
  onPageChange: (page: number) => void;
}

const Pagination: React.FC<PaginationProps> = ({ page, pageSize, totalCount, onPageChange, caption, ...props }) => {
  const t = useTranslations('pagination');

  const Component = caption ? 'div' : React.Fragment;
  const componentProps = caption ? { className: 'ml-auto' } : {};

  return (
    <Component {...componentProps}>
      <IdskPagination
        forcePage={page}
        pageCount={Math.ceil(totalCount / pageSize)}
        caption={
          typeof caption === 'boolean'
            ? t.rich('caption', {
                min: (page - 1) * pageSize + 1,
                max: Math.min(page * pageSize, totalCount),
                total: totalCount
              })
            : caption
        }
        onPageChange={({ selected }) => onPageChange(selected)}
        previousAriaLabel={t('previous')}
        nextAriaLabel={t('next')}
        ariaLabelBuilder={(selected) => `${t('page')} ${selected}`}
        {...props}
      />
    </Component>
  );
};

export default Pagination;
