'use client';

import { useMutation, useQueryClient } from '@tanstack/react-query';
import axios from 'axios';
import { useSession } from 'next-auth/react';
import { useTranslations } from 'next-intl';
import React, { useState } from 'react';
import {
  DeleteIcon,
  Table,
  TableRow,
  TableRowValue,
  TertiaryIconButton,
  CheckIcon,
  TextButton,
  AddIcon
} from '@/lib/idsk';
import { ContactType, Profile, unwrap } from '@/utils';
import { useBreakpoint, useNotifications } from '@/hooks';
import { Empty, TableSection } from '@/components/core';
import useContact from './use-contact.hook';
import ContactTableList from './ContactTableList';
import AddContactDialog from './AddContactDialog';

interface ContactProps {
  type: ContactType;
  data: string[];
  preferred: {
    email: string;
    phone: string;
  };
}

const Contact: React.FC<ContactProps> = ({ type, data, preferred }) => {
  const t = useTranslations('my_data');
  const queryClient = useQueryClient();
  const notify = useNotifications();

  const [open, setOpen] = useState<boolean>(false);

  const { data: session } = useSession();

  const { add, remove, value, error, handleChange } = useContact(session?.user, type);

  const { Breakpoints, match } = useBreakpoint();

  const view = match(Breakpoints.tb1) ? 'table' : 'list';

  const update = useMutation({
    mutationKey: ['contact'],
    mutationFn: (mutationData: any) =>
      unwrap(
        axios.put(`/api/v1/kk/profiles/${session?.user.id}`, {
          identity: {
            kid: session?.user.id,
            emailPreferred: preferred.email,
            phonePreferred: preferred.phone,
            ...mutationData
          }
        })
      ),

    onMutate: async (mutationData: any) => {
      await queryClient.cancelQueries({ queryKey: ['profile'] });
      const prevProfile: Profile | undefined = queryClient.getQueryData(['profile']);

      if (prevProfile) {
        queryClient.setQueryData(['profile'], (old: Profile) => ({
          ...old,
          identity: {
            ...old.identity,
            ...mutationData
          }
        }));
      }

      notify(t('contact_primary_success_notification'));

      return { prevProfile };
    },
    onError: (err, mutationData, context: any) => {
      queryClient.setQueryData(['profile'], context.prevProfile);

      notify({
        message: t('contact_primary_error_notification'),
        variant: 'warning'
      });
    },
    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ['profile'] });
    }
  });

  const handleRemove = (contact: string) => {
    remove.mutate({
      type: type === 'email' ? 'emails' : 'phones',
      value: contact
    });
  };

  const handleSetAsPrimary = (contact: string) => {
    update.mutate({
      [type === 'email' ? 'emailPreferred' : 'phonePreferred']: contact
    });
  };

  return (
    <>
      <AddContactDialog
        opened={open}
        type={type}
        value={value}
        error={error}
        handleChange={handleChange}
        onAdd={(primary: boolean) => {
          setOpen(false);
          add.mutate({
            type: type === 'email' ? 'emails' : 'phones',
            value
          });

          if (primary) {
            handleSetAsPrimary(value);
          }
        }}
        onClose={() => setOpen(false)}
      />
      <TableSection title={t(`contact_${type}`)} description={t(`contact_${type}_sub`)}>
        <div>
          {!data.length && <Empty title={t(`${type}_empty`)} />}
          {!!data.length && view === 'table' && (
            <Table
              headRowBordered
              headRow={
                <>
                  <TableRowValue>{t(`contact_${type}`)}</TableRowValue>
                  <TableRowValue align="right">{t(`contact_${type}_primary`)}</TableRowValue>
                </>
              }
            >
              {data.map((item) => (
                <TableRow key={item} className="flex justify-between">
                  <TableRowValue className="break-all">{item}</TableRowValue>
                  {preferred[type] === item ? (
                    <TableRowValue align="right">
                      <CheckIcon className="mx-2.5 text-alert-positive" />
                    </TableRowValue>
                  ) : (
                    <TableRowValue className="last:static" align="right">
                      <TextButton onClick={() => handleSetAsPrimary(item)}>{t('contact_set_primary')}</TextButton>
                      <TertiaryIconButton
                        variant="warning"
                        icon={<DeleteIcon />}
                        onClick={() => {
                          handleRemove(item);
                        }}
                      />
                    </TableRowValue>
                  )}
                </TableRow>
              ))}
            </Table>
          )}
          {!!data.length && view === 'list' && (
            <ContactTableList
              data={data}
              type={type}
              preferred={preferred}
              onRemove={(item) => handleRemove(item)}
              onSetPrimary={(item) => handleSetAsPrimary(item)}
            />
          )}
          <TextButton icon={<AddIcon />} className="w-fit mt-3" onClick={() => setOpen(true)}>
            {t(`add_contact_${type}`)}
          </TextButton>
        </div>
      </TableSection>
    </>
  );
};

export default Contact;
