package sk.kosice.konto.kkmessageservice.topic;

import static org.assertj.core.api.Assertions.assertThat;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.UNAUTHORIZED;
import static sk.kosice.konto.kkmessageservice.restapi.common.listing.ListOrderingParser.ASCENDING;
import static sk.kosice.konto.kkmessageservice.restapi.common.listing.ListOrderingParser.DESCENDING;
import static sk.kosice.konto.kkmessageservice.topic.TopicApiFixtures.createTopicRequest;

import java.util.List;
import java.util.UUID;
import org.junit.jupiter.api.Test;
import org.springframework.http.HttpStatus;
import sk.kosice.konto.kkmessageservice.domain.common.listing.TopicListingAttribute;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.topic.TopicDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.topic.TopicListResponse;

public class TopicListingFeatureSpec extends TopicFeatureSpec {

  @Test
  public void thatEmptyTopicListCanBeRetrieved() {
    final TopicListResponse messages = listTopics(UUID.randomUUID());
    assertThat(messages.items()).isEmpty();
    assertThat(messages.totalCount()).isZero();
  }

  @Test
  public void thatTopicListCanBeRetrievedByRecipientKid() {
    createTopicsForFilteringAndSorting();

    final TopicListResponse messages = listTopics(ORGANIZATION_ID_1);
    assertThat(messages.items()).isNotEmpty();
    assertThat(messages.totalCount()).isEqualTo(3);
  }

  @Test
  public void thatTopicListCannotBeRetrievedBWithoutToken() {
    createTopicsForFilteringAndSorting();

    final ErrorDetailResponse errorDetailResponse =
        listTopicsWithoutJwt(ORGANIZATION_ID_1, HttpStatus.UNAUTHORIZED, ErrorDetailResponse.class);
    assertThat(errorDetailResponse.primaryFault().get().faultCode())
        .isNotNull()
        .isEqualTo(UNAUTHORIZED.toString());
  }

  @Test
  public void thatTopicListCanBeFilteredByName() {
    final List<TopicDetailResponse> topics = createTopicsForFilteringAndSorting();

    final String filter =
        String.format(
            "?filter=%s==%s", TopicListingAttribute.NAME.apiName(), topics.get(0).getName());

    checkFiltering(ORGANIZATION_ID_1, filter, 1, topics.get(0));
  }

  @Test
  public void thatTopicListCanBeFilteredByDescription() {
    final List<TopicDetailResponse> topics = createTopicsForFilteringAndSorting();

    final String filter =
        String.format(
            "?filter=%s==%s",
            TopicListingAttribute.DESCRIPTION.apiName(), topics.get(0).getDescription());

    checkFiltering(ORGANIZATION_ID_1, filter, 1, topics.get(0));
  }

  @Test
  public void thatTopicListCanBeSortedByName() {
    final List<TopicDetailResponse> messages = createTopicsForFilteringAndSorting();

    checkSorting(
        ORGANIZATION_ID_1,
        TopicListingAttribute.NAME.apiName() + ASCENDING,
        messages.size(),
        messages.get(0),
        messages.get(1),
        messages.get(2));

    checkSorting(
        ORGANIZATION_ID_1,
        TopicListingAttribute.NAME.apiName() + DESCENDING,
        messages.size(),
        messages.get(2),
        messages.get(1),
        messages.get(0));
  }

  private List<TopicDetailResponse> createTopicsForFilteringAndSorting() {

    final TopicDetailResponse topic1 =
        createTopic(
            ORGANIZATION_ID_1, createTopicRequest().withName("name1").withDescription("desc1"));

    final TopicDetailResponse topic2 =
        createTopic(
            ORGANIZATION_ID_1, createTopicRequest().withName("name2").withDescription("desc2"));

    final TopicDetailResponse topic3 =
        createTopic(
            ORGANIZATION_ID_1, createTopicRequest().withName("name3").withDescription("desc3"));

    return List.of(topic1, topic2, topic3);
  }
}
