package sk.kosice.konto.kknotificationservice.restapi.dto.common.error;

import static io.swagger.v3.oas.annotations.media.Schema.RequiredMode.REQUIRED;

import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import io.swagger.v3.oas.annotations.media.ArraySchema;
import io.swagger.v3.oas.annotations.media.Schema;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import org.immutables.value.Value;

@JsonDeserialize(as = ImmutableErrorDetailResponse.class)
@JsonSerialize(as = ImmutableErrorDetailResponse.class)
@Value.Immutable
public interface ErrorDetailResponse {

  @Schema(
      description =
          """
        Unique identifier of the given transaction in internal system to identify a whole flow of requests, calls or steps in internal microservices in case of any issues.</br>
        32 character lowercase hex String.

        See OpenTelemetry doc: <a href="https://www.w3.org/TR/trace-context/#trace-id">trace-id</a>
        """,
      type = "string",
      format = "base16",
      requiredMode = REQUIRED,
      example = "4bf92f3577b34da6a3ce929d0e0e4736")
  String correlationId();

  @ArraySchema(
      arraySchema =
          @Schema(
              description = "A list of faults - particular errors.",
              requiredMode = REQUIRED,
              type = "array",
              example =
                  """
        [
           {
             "serviceId": "123e4567-e89b-12d3-a456-426614174007",
             "faultCode": "PAYMENT_FAILED",
             "faultMessage": "Payment with id %s failed.",
             "faultMessageParams": ["d1c49ac3-6d1c-4d7d-955b-2f5863fb8345"]
           },
           {
             "serviceId": "123e4567-e89b-12d3-a456-426614174008",
             "faultCode": "PERMISSION_DENIED",
             "faultMessage": "Current user %s have no valid permission to execute operation",
             "faultMessageParams": ["d1c49ac3-6d1c-4d7d-955b-2f5863fb8345"]
           }
        ]
        """))
  List<ErrorDetailResponseItem> fault();

  default Optional<ErrorDetailResponseItem> primaryFault() {
    return Optional.ofNullable(fault()).stream().flatMap(Collection::stream).findFirst();
  }
}
