'use client';

import React, { useMemo, useState } from 'react';
import { useTranslations } from 'next-intl';
import axios from 'axios';
import { useSession } from 'next-auth/react';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { Loader, TextButton, AddIcon, TertiaryIconButton, DeleteIcon, Tooltip, InfoIcon } from '@/lib/idsk';
import { AddAddressDialog, Card, Data, DataItem, DeleteAddressDialog } from '@/components';
import { address as buildAddress, Address, Profile, unwrap, RequestError } from '@/utils';
import { useNotifications } from '@/hooks';

type IAddress = Address & { address?: string };

interface AddressesProps {
  status: 'pending' | 'success' | 'error';
  data: Address[];
}

interface AddressItemProps {
  label: string;
  address: IAddress | null;
  onAdd?: () => void;
  onDelete?: (address: IAddress) => void;
}

const AddressItem: React.FC<AddressItemProps> = ({ address, label, onAdd, onDelete }) => {
  const t = useTranslations('my_data');

  return (
    <DataItem
      className="tb2:flex-row tb2:gap-4 gap-1 flex-col"
      labelClassName="w-full mb:w-full tb2:w-1/3"
      childrenClassName="w-full mb:w-full tb2:w-2/3"
      label={label}
    >
      {address?.address ? (
        <span className="flex justify-between items-center relative w-full">
          {address.address}{' '}
          {!!onDelete && (
            <TertiaryIconButton
              className="tb2:absolute right-0"
              icon={<DeleteIcon />}
              variant="warning"
              onClick={() => onDelete(address)}
            />
          )}
        </span>
      ) : (
        !!onAdd && (
          <TextButton className="-ml-4" icon={<AddIcon />} onClick={onAdd}>
            {t('add_address')}
          </TextButton>
        )
      )}
    </DataItem>
  );
};

const Addresses: React.FC<AddressesProps> = ({ status, data }) => {
  const [addAddressType, setAddAddressType] = useState<'permanent' | 'delivery' | 'temporary' | null>(null);
  const [addrToDelete, setAddrToDelete] = useState<Address | null>(null);

  const t = useTranslations('my_data');
  const session = useSession();
  const notify = useNotifications();
  const queryClient = useQueryClient();

  const { mutate } = useMutation<unknown, RequestError>({
    mutationKey: ['address'],
    mutationFn: (mutationData: any) =>
      unwrap(axios.delete(`/api/v1/kk/profiles/${mutationData.userId}/address/${mutationData.address.aid}`)),
    onMutate: async (mutationData: any) => {
      notify({ message: t('notif_address_deleted') });

      await queryClient.cancelQueries({ queryKey: ['profile'] });

      const prevProfile: Profile | undefined = queryClient.getQueryData(['profile']);

      if (prevProfile) {
        const newAddresses = prevProfile.addresses.filter((address) => address.aid !== mutationData.address.aid);

        queryClient.setQueryData(['profile'], (old: Profile) => ({
          ...old,
          addresses: newAddresses
        }));
      }

      return { prevProfile };
    },
    onError: (err, mutationData, context: any) => {
      queryClient.setQueryData(['profile'], context.prevProfile);

      notify({
        message: 'Nepodarilo sa zmazat adresu',
        variant: 'warning'
      });
    },
    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ['profile'] });
    }
  });

  const addresses = useMemo(() => {
    const result: Record<string, IAddress | null> = {
      permanent: null,
      delivery: null,
      temporary: null
    };

    if (status === 'success' && data) {
      data.forEach((address: Address) => {
        if (address.type === 'permanent') {
          result.permanent = address;
          result.permanent.address = buildAddress(address);
        } else if (address.type === 'delivery') {
          result.delivery = address;
          result.delivery.address = buildAddress(address);
        } else if (address.type === 'temporary') {
          result.temporary = address;
          result.temporary.address = buildAddress(address);
        }
      });
    }

    return result;
  }, [status, data]);

  const handleAddAddress = (type: 'permanent' | 'delivery' | 'temporary') => {
    setAddAddressType(type);
  };

  const handleDeleteAddress = async () => {
    const address = addrToDelete as Address;
    setAddrToDelete(null);

    mutate({
      userId: session.data?.user.id ?? '',
      address
    } as any);
  };

  return (
    <>
      <AddAddressDialog type={addAddressType} toggleOpened={() => setAddAddressType(null)} />

      <DeleteAddressDialog
        address={addrToDelete}
        onSubmit={handleDeleteAddress}
        onCancel={() => setAddrToDelete(null)}
      />

      <Card
        title={t('addresses')}
        icon={
          <Tooltip tooltip={t('address_tooltip')}>
            <InfoIcon className="text-primary-medium" width="1.5em" height="1.5em" />
          </Tooltip>
        }
      >
        {status === 'pending' && (
          <div className="px-6 py-12 flex items-center justify-center w-full">
            <Loader />
          </div>
        )}
        {status === 'success' && (
          <Data>
            <AddressItem
              address={addresses.permanent}
              label={t('permanent_address')}
              onAdd={() => handleAddAddress('permanent')}
            />
            <AddressItem
              address={addresses.delivery}
              label={t('delivery_address')}
              onAdd={() => handleAddAddress('delivery')}
              onDelete={() => setAddrToDelete(addresses.delivery)}
            />
            <AddressItem
              address={addresses.temporary}
              label={t('temporary_address')}
              onAdd={() => handleAddAddress('temporary')}
              onDelete={() => setAddrToDelete(addresses.temporary)}
            />
          </Data>
        )}
      </Card>
    </>
  );
};

export default Addresses;
