package sk.kosice.konto.kkmessageservice.repository.rsql.converter;

import static java.time.format.DateTimeFormatter.ISO_LOCAL_DATE;
import static java.time.format.DateTimeFormatter.ISO_LOCAL_TIME;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalQueries;
import sk.kosice.konto.kkmessageservice.repository.rsql.RsqlErrorCode;

public class LocalDateValueConverter implements FieldValueConverter<LocalDate> {

  @Override
  public boolean isAccessibleFor(Class clazz) {
    return LocalDate.class.isAssignableFrom(clazz);
  }

  private static final DateTimeFormatter ISO_LOCAL_DATE_OPTIONAL_TIME =
      new DateTimeFormatterBuilder()
          .parseCaseInsensitive()
          .append(ISO_LOCAL_DATE)
          .optionalStart()
          .appendLiteral('T')
          .append(ISO_LOCAL_TIME)
          .optionalEnd()
          .optionalStart()
          .appendOffsetId()
          .optionalEnd()
          .optionalStart()
          .appendLiteral("[")
          .appendZoneRegionId()
          .appendLiteral("]")
          .optionalEnd()
          .toFormatter();

  public LocalDate from(String from) {
    TemporalAccessor parsed;
    try {
      parsed = ISO_LOCAL_DATE_OPTIONAL_TIME.parse(from);
    } catch (DateTimeParseException e) {
      throw RsqlErrorCode.INVALID_RSQL_DATETIME_VALUE_EXCEPTION
          .createError(from)
          .convertToException();
    }
    final LocalDate date = parsed.query(TemporalQueries.localDate());
    final LocalTime time = parsed.query(TemporalQueries.localTime());
    final ZoneId zone = parsed.query(TemporalQueries.zone());

    if (time != null && zone != null) {
      return LocalDateTime.of(date, time)
          .atZone(zone)
          .withZoneSameInstant(ZoneId.of("UTC"))
          .toLocalDate();
    } else if (date != null) {
      return LocalDate.of(date.getYear(), date.getMonth(), date.getDayOfMonth());
    }
    throw RsqlErrorCode.INVALID_RSQL_DATETIME_VALUE_EXCEPTION
        .createError(from)
        .convertToException();
  }
}
