package sk.kosice.konto.kkmessageservice.repository.support.transactional;

import jakarta.inject.Named;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionTemplate;
import sk.kosice.konto.kkmessageservice.domain.common.error.BusinessException;
import sk.kosice.konto.kkmessageservice.domain.common.transactional.Transactional;

@Named
public class SpringTransactional implements Transactional {

  private static final Logger log = LoggerFactory.getLogger(SpringTransactional.class);

  private final TransactionTemplate transactionTemplate;

  @Autowired
  public SpringTransactional(PlatformTransactionManager transactionManager) {
    this.transactionTemplate = new TransactionTemplate(transactionManager);
  }

  @Override
  public <V> V result(Supplier<V> transactionalCall) {
    return result(transactionalCall, () -> null);
  }

  @Override
  public <V> V result(Supplier<V> transactionalCall, Supplier<V> rollbackCall) {
    final AtomicReference<V> savedResult = new AtomicReference<>();
    try {
      log.info("Starting transactional call");
      transactionTemplate.execute(
          transactionStatus -> {
            try {
              final V result = transactionalCall.get();
              savedResult.set(result);
              log.debug("Call returned valid result - committing");
              return result;
            } catch (BusinessException e) {
              log.debug("Call returned invalid result - rolling back");
              throw e;
            }
          });
      log.info("Transaction committed");
      return savedResult.get();
    } catch (BusinessException be) {
      log.info("Starting rollback call");
      transactionTemplate.execute(
          transactionStatus -> {
            try {
              rollbackCall.get();
              log.debug("Rollback call returned valid result - committing");
              return null;
            } catch (BusinessException e) {
              log.debug("Rollback call returned invalid result - rolling back");
              throw e;
            }
          });
      throw be;
    } catch (Throwable e) {
      log.info("Transaction rolled back");
      throw e;
    }
  }
}
