'use client';

import axios from 'axios';
import React, { useState } from 'react';
import { api, NewTopic, unwrap } from '@/utils';
import { Dialog, Input, Loader, PrimaryButton, SecondaryButton, TextField } from '@/lib/idsk';
import { useMutation } from '@tanstack/react-query';
import { useNotifications } from '@/hooks';

interface CreateTopicDialogProps {
  organizationId: string;
  onCreate: () => void;
  children: React.ReactElement<{ onClick: () => void }>;
}

const CreateTopicDialog: React.FC<CreateTopicDialogProps> = ({ children, organizationId, onCreate }) => {
  const [isOpen, setIsOpen] = useState<boolean>(false);

  const notify = useNotifications();

  const mutation = useMutation({
    mutationKey: ['topics'],
    mutationFn: (data: NewTopic) => unwrap(axios.post(`/api${api.topics(organizationId)}`, data)),
    onSuccess: () => {
      setIsOpen(false);
      onCreate();
    },
    onError: () => {
      setIsOpen(false);
      notify({
        message: 'Tému sa nepodarilo vytvoriť.',
        variant: 'warning'
      });
    }
  });

  const createTopic = (formData: FormData) => {
    const name = formData.get('name') as string;
    const description = formData.get('description') as string;
    mutation.mutate({
      name,
      description
    });
  };

  const isLoading = mutation.isPending;

  return (
    <>
      {React.isValidElement(children) && React.cloneElement(children, { onClick: () => setIsOpen(true) })}
      {isOpen && (
        <form action={createTopic}>
          <Dialog
            opened={isOpen}
            toggleOpened={() => setIsOpen(false)}
            title="Vytvoriť tému"
            primaryButton={<PrimaryButton disabled={isLoading}>Vytvoriť</PrimaryButton>}
            secondaryButton={<SecondaryButton onClick={() => setIsOpen(false)}>Zatvoriť</SecondaryButton>}
          >
            <div className="w-full tb2:min-w-[42rem] flex flex-col justify-start items-start relative">
              <Input label="Názov" name="name" required mandatory fullWidth />
              <TextField label="Popis" name="description" required mandatory fullWidth />

              {isLoading && (
                <div className="bg-white flex justify-center items-center absolute top-0 left-0 w-full h-full">
                  <Loader />
                </div>
              )}
            </div>
          </Dialog>
        </form>
      )}
    </>
  );
};

export default CreateTopicDialog;
