package sk.kosice.konto.kkmessageservice.repository.topic;

import static sk.kosice.konto.kkmessageservice.domain.topic.error.TopicErrorCode.TOPIC_DOES_NOT_EXIST;
import static sk.kosice.konto.kkmessageservice.repository.model.Tables.TOPIC;
import static sk.kosice.konto.kkmessageservice.repository.topic.mapper.JooqTopicRepositoryMapper.listingMapper;
import static sk.kosice.konto.kkmessageservice.repository.topic.mapper.JooqTopicRepositoryMapper.topicRecordMapper;

import java.util.List;
import java.util.UUID;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.SelectConditionStep;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import sk.kosice.konto.kkmessageservice.domain.common.error.BusinessException;
import sk.kosice.konto.kkmessageservice.domain.topic.data.DeleteTopicData;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.ListOfTopics;
import sk.kosice.konto.kkmessageservice.domain.topic.entity.TopicEntity;
import sk.kosice.konto.kkmessageservice.domain.topic.query.FindTopicByIdQuery;
import sk.kosice.konto.kkmessageservice.domain.topic.query.TopicListingQuery;
import sk.kosice.konto.kkmessageservice.repository.JooqRepository;
import sk.kosice.konto.kkmessageservice.repository.provider.TopicJooqRsqlMetadataConfigProvider;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.handler.JooqPaginatedSelectQueryHandler;
import sk.kosice.konto.kkmessageservice.repository.rsql.jooq.handler.mapper.PaginatedResultMapper;
import sk.kosice.konto.kkmessageservice.repository.topic.error.constraint.TopicConstraintErrorCode;

@Repository
public class JooqTopicRepository extends JooqRepository implements TopicRepositoryAdapter {

  @Autowired
  protected JooqTopicRepository(DSLContext dslContext) {
    super(dslContext);
  }

  @Override
  public void insert(TopicEntity data) throws BusinessException {
    log.info("Applying {}", data);

    handleInsertResult(
        () ->
            dslContext
                .insertInto(TOPIC)
                .set(TOPIC.ID, data.id())
                .set(TOPIC.NAME, data.name())
                .set(TOPIC.DESCRIPTION, data.description())
                .set(TOPIC.ORGANIZATION_ID, data.organizationId())
                .set(TOPIC.ORGANIZATION_NAME, data.organizationName())
                .execute(),
        data,
        TopicConstraintErrorCode.values());
  }

  @Override
  public void update(TopicEntity data) throws BusinessException {
    log.info("Applying {}", data);

    handleUpdateResult(
        () ->
            dslContext
                .update(TOPIC)
                .set(TOPIC.NAME, data.name())
                .set(TOPIC.DESCRIPTION, data.description())
                .set(TOPIC.ORGANIZATION_NAME, data.organizationName())
                .where(TOPIC.ID.eq(data.id()))
                .and(TOPIC.ORGANIZATION_ID.eq(data.organizationId()))
                .execute(),
        data,
        TopicConstraintErrorCode.values());
  }

  @Override
  public void delete(DeleteTopicData data) throws BusinessException {
    log.info("Applying {}", data);

    handleDeleteResult(
        () ->
            dslContext
                .deleteFrom(TOPIC)
                .where(TOPIC.ORGANIZATION_ID.eq(data.organizationId()))
                .and(TOPIC.ID.eq(data.id()))
                .execute(),
        data,
        TopicConstraintErrorCode.values());
  }

  @Override
  public TopicEntity findOne(FindTopicByIdQuery query) throws BusinessException {
    log.info("Applying {}", query);

    return handleFindOneResult(
        () ->
            dslContext
                .selectFrom(TOPIC)
                .where(TOPIC.ID.eq(query.id()))
                .and(TOPIC.ORGANIZATION_ID.eq(query.organizationId()))
                .fetch(topicRecordMapper()),
        query,
        TopicConstraintErrorCode.values());
  }

  @Override
  public ListOfTopics list(TopicListingQuery query) throws BusinessException {
    final SelectConditionStep<Record> select =
        dslContext.select().from(TOPIC).where(TOPIC.ORGANIZATION_ID.eq(query.organizationId()));

    final JooqPaginatedSelectQueryHandler<TopicListingQuery, ListOfTopics> paginatedListingHandler =
        new JooqPaginatedSelectQueryHandler<>(query, new TopicJooqRsqlMetadataConfigProvider());

    final PaginatedResultMapper<TopicListingQuery, ListOfTopics> paginated =
        paginatedListingHandler.handle(select.getQuery());

    return paginated.map(listingMapper);
  }

  @Override
  public List<TopicEntity> listByOrganizationId(UUID organizationId) {
    log.info("Applying findOneByOrganizationId {}", organizationId);

    return dslContext
        .selectFrom(TOPIC)
        .where(TOPIC.ORGANIZATION_ID.eq(organizationId))
        .fetch(topicRecordMapper());
  }

  @Override
  protected void createEntityDoesNotExistError(Object query) {
    switch (query) {
      case FindTopicByIdQuery findTopicByIdQuery:
        throw TOPIC_DOES_NOT_EXIST
            .createError(findTopicByIdQuery.id().toString())
            .convertToException();
      default:
        throw TOPIC_DOES_NOT_EXIST.createError(query).convertToException();
    }
  }
}
