package sk.kosice.konto.kkmessageservice;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.doThrow;
import static sk.kosice.konto.kkmessageservice.domain.organization.error.OrganizationErrorCode.ORGANIZATION_DOES_NOT_EXIST;
import static sk.kosice.konto.kkmessageservice.restapi.controller.BaseController.EMPLOYEE_ID_CLAIM_NAME;

import io.restassured.RestAssured;
import io.restassured.http.ContentType;
import io.restassured.http.Header;
import io.restassured.http.Headers;
import io.restassured.specification.RequestSpecification;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;
import no.nav.security.mock.oauth2.MockOAuth2Server;
import org.jooq.DSLContext;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.boot.test.web.server.LocalServerPort;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;
import sk.kosice.konto.kkmessageservice.business.organization.port.outbound.QueryOrganizationPort;
import sk.kosice.konto.kkmessageservice.business.topic.port.inbound.SynchronizeTopicOrganizationNameUseCase;
import sk.kosice.konto.kkmessageservice.domain.organization.entity.ImmutableOrganizationEntity;
import sk.kosice.konto.kkmessageservice.domain.organization.entity.OrganizationEntity;
import sk.kosice.konto.kkmessageservice.repository.message.JooqMessageRepository;
import sk.kosice.konto.kkmessageservice.repository.model.Tables;
import sk.kosice.konto.kkmessageservice.repository.recipient.JooqRecipientRepository;
import sk.kosice.konto.kkmessageservice.repository.subscription.JooqSubscriptionRepository;
import sk.kosice.konto.kkmessageservice.repository.topic.JooqTopicRepository;

@ExtendWith(SpringExtension.class)
@SpringBootTest(
    webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT,
    classes = KkMessageServiceApplication.class)
@ActiveProfiles("test")
public abstract class ServiceAppFeatureSpec {

  @LocalServerPort protected int actualPort;

  @Autowired protected DSLContext dslContext;
  @Autowired protected JooqMessageRepository messageRepository;
  @Autowired protected JooqRecipientRepository recipientRepository;
  @Autowired protected JooqTopicRepository topicRepository;
  @Autowired protected JooqSubscriptionRepository subscriptionRepository;

  @Autowired
  protected SynchronizeTopicOrganizationNameUseCase synchronizeTopicOrganizationNameUseCase;

  @MockBean QueryOrganizationPort queryOrganizationPort;

  protected static MockOAuth2Server oAuth2Server;
  protected static String EXTENSION_KID_CLAIM;
  protected static String EMPLOYEE_JWT_TOKEN;
  protected static String CITIZEN_JWT_TOKEN;
  protected static String APP_INTEGRATION_JWT_TOKEN;
  protected static String EMPLOYEE_SUB_CLAIM;

  // Role: Admin
  protected static final UUID ROLE_ADMIN_ID =
      UUID.fromString("58aebdd7-726a-4f0a-9841-d4cb1c00b4d1");
  // Role: Global Admin
  protected static final UUID ROLE_GLOBAL_ADMIN_ID =
      UUID.fromString("496a4953-e32c-44ab-9fc5-d730f4acd617");
  protected static final UUID ORGANIZATION_ID_1 =
      UUID.fromString("b96a4953-e32c-44ab-9fc5-d730f4acd61a");
  protected static final UUID ORGANIZATION_ID_2 =
      UUID.fromString("1fb89e46-ce51-4998-af74-0883c65d7af6");
  protected static final UUID ORGANIZATION_ID_3 =
      UUID.fromString("a5132345-ad3e-46cd-b7cf-b500ce175931");
  protected static final UUID NOT_EXISTING_ORGANIZATION_ID =
      UUID.fromString("58564c74-ff15-497a-82eb-2caff2ef2add");

  public void prepareInvalidPermissions() {
    doReturn(List.of()).when(queryOrganizationPort).getUserGroups(EMPLOYEE_SUB_CLAIM);
    doReturn(List.of()).when(queryOrganizationPort).getUserGroupIds(EMPLOYEE_SUB_CLAIM);
  }

  @BeforeEach
  public void preparePermissions() {
    List<OrganizationEntity> listOfRoleOrganizations =
        new ArrayList<>() {
          {
            add(
                ImmutableOrganizationEntity.builder()
                    .id(ROLE_ADMIN_ID)
                    .isCityDistrict(false)
                    .name("Admin")
                    .build());
            add(
                ImmutableOrganizationEntity.builder()
                    .id(ROLE_GLOBAL_ADMIN_ID)
                    .isCityDistrict(false)
                    .name("Admin-Global")
                    .build());
            addAll(buildOrganizationEntities());
          }
        };

    doReturn(listOfRoleOrganizations).when(queryOrganizationPort).getUserGroups(EMPLOYEE_SUB_CLAIM);
    doReturn(
            listOfRoleOrganizations.parallelStream()
                .map(OrganizationEntity::id)
                .collect(Collectors.toList()))
        .when(queryOrganizationPort)
        .getUserGroupIds(EMPLOYEE_SUB_CLAIM);
  }

  public void prepareUpdatedOrganizationListResponse() {
    doReturn(buildOrganizationEntities()).when(queryOrganizationPort).list(any());
  }

  private List<OrganizationEntity> buildOrganizationEntities() {
    final var organization1 =
        ImmutableOrganizationEntity.builder()
            .id(ORGANIZATION_ID_1)
            .isCityDistrict(true)
            .name("updatedOrganizationName1")
            .build();

    final var organization2 =
        ImmutableOrganizationEntity.builder()
            .id(ORGANIZATION_ID_2)
            .isCityDistrict(false)
            .name("updatedOrganizationName2")
            .build();

    final var organization3 =
        ImmutableOrganizationEntity.builder()
            .id(ORGANIZATION_ID_3)
            .isCityDistrict(false)
            .name(UUID.randomUUID().toString())
            .build();

    return List.of(organization1, organization2, organization3);
  }

  @BeforeEach
  public void prepareOrganizationFindOneResponse() {
    final var organization1 =
        ImmutableOrganizationEntity.builder()
            .id(ORGANIZATION_ID_1)
            .isCityDistrict(true)
            .name("organizationName1")
            .build();

    final var organization2 =
        ImmutableOrganizationEntity.builder()
            .id(ORGANIZATION_ID_2)
            .isCityDistrict(false)
            .name("organizationName2")
            .build();

    doReturn(organization1).when(queryOrganizationPort).findOne(ORGANIZATION_ID_1);
    doReturn(organization2).when(queryOrganizationPort).findOne(ORGANIZATION_ID_2);
    doThrow(
            ORGANIZATION_DOES_NOT_EXIST
                .createError(NOT_EXISTING_ORGANIZATION_ID)
                .convertToException())
        .when(queryOrganizationPort)
        .findOne(NOT_EXISTING_ORGANIZATION_ID);
  }

  public void prepareEmptyOrganizationListResponse() {
    doReturn(List.of()).when(queryOrganizationPort).list(any());
  }

  @BeforeEach
  public void prepareOrganizationListResponse() {
    final var organization1 =
        ImmutableOrganizationEntity.builder()
            .id(ORGANIZATION_ID_1)
            .isCityDistrict(true)
            .name("organizationName1")
            .build();

    final var organization2 =
        ImmutableOrganizationEntity.builder()
            .id(ORGANIZATION_ID_2)
            .isCityDistrict(false)
            .name("organizationName2")
            .build();

    final var organization3 =
        ImmutableOrganizationEntity.builder()
            .id(UUID.randomUUID())
            .isCityDistrict(false)
            .name(UUID.randomUUID().toString())
            .build();

    doReturn(List.of(organization1, organization2, organization3))
        .when(queryOrganizationPort)
        .list(any());
  }

  @BeforeAll
  public static void mockOAuthStart() throws UnknownHostException {
    oAuth2Server = new MockOAuth2Server();
    oAuth2Server.start(InetAddress.getByName("localhost"), 8081);
    EXTENSION_KID_CLAIM = UUID.randomUUID().toString();
    EMPLOYEE_SUB_CLAIM = UUID.randomUUID().toString();
    EMPLOYEE_JWT_TOKEN =
        oAuth2Server
            .issueToken(
                "employee",
                EMPLOYEE_SUB_CLAIM,
                null,
                Map.of(EMPLOYEE_ID_CLAIM_NAME, EMPLOYEE_SUB_CLAIM))
            .serialize();

    CITIZEN_JWT_TOKEN =
        oAuth2Server
            .issueToken("citizen", "citizenSub", null, Map.of("extension_kid", EXTENSION_KID_CLAIM))
            .serialize();
    APP_INTEGRATION_JWT_TOKEN =
        oAuth2Server
            .issueToken(
                "app-integration",
                EMPLOYEE_SUB_CLAIM,
                null,
                Map.of(EMPLOYEE_ID_CLAIM_NAME, EMPLOYEE_SUB_CLAIM))
            .serialize();
  }

  @AfterAll
  public static void mockOAuthStop() {
    oAuth2Server.shutdown();
  }

  @BeforeEach
  public void prepareDb() {
    dslContext.deleteFrom(Tables.MESSAGE).execute();
    dslContext.deleteFrom(Tables.SUBSCRIPTION).execute();
    dslContext.deleteFrom(Tables.RECIPIENT).execute();
    dslContext.deleteFrom(Tables.TOPIC).execute();
  }

  public static Headers defaultHeaders() {
    return new Headers(
        List.of(
            new Header("correlationId", UUID.randomUUID().toString()),
            new Header("X-APP-ID", UUID.randomUUID().toString()),
            new Header("X-APP-VERSION", "1.0.0"),
            new Header("X-APP-PLATFORM", "SERVER")));
  }

  public static Headers defaultHeadersWithOnBehalfOf() {
    return new Headers(
        List.of(
            new Header("correlationId", UUID.randomUUID().toString()),
            new Header("X-APP-ID", UUID.randomUUID().toString()),
            new Header("X-APP-VERSION", "1.0.0"),
            new Header("X-APP-PLATFORM", "SERVER"),
            new Header("X-On-Behalf-Of", ORGANIZATION_ID_1.toString())));
  }

  protected RequestSpecification requestSpecification() {
    return RestAssured.given()
        .log()
        .all()
        .port(actualPort)
        .accept(ContentType.JSON)
        .contentType(ContentType.JSON)
        .headers(defaultHeaders());
  }
}
