package sk.kosice.konto.kkmessageservice.message;

import static org.assertj.core.api.AssertionsForClassTypes.assertThat;
import static sk.kosice.konto.kkmessageservice.domain.common.enumeration.Location.KOSICE_BARCA;
import static sk.kosice.konto.kkmessageservice.domain.common.enumeration.Location.KOSICE_DZUNGLA;
import static sk.kosice.konto.kkmessageservice.domain.common.enumeration.Location.KOSICE_STARO_MESTO;
import static sk.kosice.konto.kkmessageservice.domain.permission.error.PermissionErrorCode.ACTION_IS_FORBIDDEN;
import static sk.kosice.konto.kkmessageservice.message.MessageApiFixtures.createMessageRequest;
import static sk.kosice.konto.kkmessageservice.message.MessageApiFixtures.createMessageRequestWithActions;
import static sk.kosice.konto.kkmessageservice.restapi.common.error.BaseApiErrorCode.UNAUTHORIZED;

import java.util.UUID;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.springframework.http.HttpStatus;
import sk.kosice.konto.kkmessageservice.restapi.dto.common.error.ErrorDetailResponse;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageCreateRequest;
import sk.kosice.konto.kkmessageservice.restapi.dto.message.MessageDetailResponse;

public class MessageCreateFeatureSpec extends MessageFeatureSpec {

  @Test
  public void thatMessageCanBeCreatedWithEmployeeToken() {
    final UUID recipientKid =
        persistAndFindRecipient(recipientRepository, prepareRecipientData(EXTENSION_KID_CLAIM))
            .kid();
    final UUID topicId = persistAndFindTopic(topicRepository, ORGANIZATION_ID_1).id();
    final MessageCreateRequest request = createMessageRequestWithActions(topicId);
    final MessageDetailResponse detailResponse = createMessage(recipientKid, request);

    assertThat(detailResponse.getTitle()).isEqualTo(request.getTitle());
    assertThat(detailResponse.getTags()).isEqualTo(request.getTags());
    assertThat(detailResponse.getBody()).isEqualTo(request.getBody());
    assertThat(detailResponse.getBodyShort()).isEqualTo(request.getBodyShort());
    assertThat(detailResponse.getBodyType()).isEqualTo(request.getBodyType());
    assertThat(detailResponse.getSenderName()).isEqualTo("organizationName1");
    assertThat(detailResponse.getLocations().size()).isEqualTo(3);
    assertThat(detailResponse.getLocations().get(0)).isEqualTo(KOSICE_BARCA);
    assertThat(detailResponse.getLocations().get(1)).isEqualTo(KOSICE_STARO_MESTO);
    assertThat(detailResponse.getLocations().get(2)).isEqualTo(KOSICE_DZUNGLA);
  }

  @Test
  public void thatMessageCanBeCreatedWithAppIntegrationToken() {
    final UUID recipientKid =
        persistAndFindRecipient(recipientRepository, prepareRecipientData(EXTENSION_KID_CLAIM))
            .kid();
    final UUID topicId = persistAndFindTopic(topicRepository, ORGANIZATION_ID_1).id();
    final MessageCreateRequest request = createMessageRequestWithActions(topicId);
    final MessageDetailResponse detailResponse = createMessageAppIntegration(recipientKid, request);

    assertThat(detailResponse.getTitle()).isEqualTo(request.getTitle());
    assertThat(detailResponse.getTags()).isEqualTo(request.getTags());
    assertThat(detailResponse.getBody()).isEqualTo(request.getBody());
  }

  @Test
  public void thatMessageCanBeCreatedWithoutToken() {
    final UUID recipientKid =
        persistAndFindRecipient(recipientRepository, prepareRecipientData(EXTENSION_KID_CLAIM))
            .kid();
    final UUID topicId = persistAndFindTopic(topicRepository, ORGANIZATION_ID_1).id();
    final MessageCreateRequest request = createMessageRequestWithActions(topicId);
    final ErrorDetailResponse detailResponse = createMessageWithoutJwt(recipientKid, request);

    assertThat(detailResponse.primaryFault().get().faultCode())
        .isNotNull()
        .isEqualTo(UNAUTHORIZED.toString());
  }

  @Disabled
  @Test
  public void thatMessageCannotBeCreatedWithNotExistingRecipientKid() {
    final UUID recipientKid = UUID.randomUUID();
    final ErrorDetailResponse errorResponse =
        createMessage(
            recipientKid,
            createMessageRequest(UUID.randomUUID()),
            HttpStatus.NOT_FOUND,
            ErrorDetailResponse.class);

    assertThat(errorResponse).isNotNull();
    // assertThat(errorResponse.primaryFault().get().faultCode()).isEqualTo(RECIPIENT_DOES_NOT_EXIST.toString());
    assertThat(errorResponse.correlationId()).isNotEmpty();
    assertThat(
            errorResponse
                .primaryFault()
                .get()
                .faultMessageParams()
                .contains(recipientKid.toString()))
        .isTrue();
  }

  @Test
  public void thatPublicMessageCanBeCreated() {
    final UUID topicId = persistAndFindTopic(topicRepository, ORGANIZATION_ID_1).id();
    final MessageCreateRequest request = createMessageRequest(topicId);
    final MessageDetailResponse detailResponse = createPublicMessage(ORGANIZATION_ID_1, request);

    assertThat(detailResponse.getTitle()).isEqualTo(request.getTitle());
    assertThat(detailResponse.getTags()).isEqualTo(request.getTags());
    assertThat(detailResponse.getBody()).isEqualTo(request.getBody());
    assertThat(detailResponse.getSenderName()).isEqualTo("organizationName1");
  }

  @Test
  public void thatMessageCannotBeCreatedWIthEmployeeTokenWithoutPermissions() {
    prepareInvalidPermissions();
    final UUID recipientKid =
        persistAndFindRecipient(recipientRepository, prepareRecipientData(EXTENSION_KID_CLAIM))
            .kid();
    final UUID topicId = persistAndFindTopic(topicRepository, ORGANIZATION_ID_1).id();
    final MessageCreateRequest request = createMessageRequestWithActions(topicId);
    final ErrorDetailResponse detailResponse =
        createMessage(recipientKid, request, HttpStatus.FORBIDDEN, ErrorDetailResponse.class);

    assertThat(detailResponse.primaryFault().get().faultCode())
        .isEqualTo(ACTION_IS_FORBIDDEN.toString());
  }
}
