import axios, { isAxiosError } from 'axios';
import { redirect } from 'next/navigation';
import { NextRequest } from 'next/server';
import { providerCache, getPolicy } from '@/lib/auth/credentials';
import { signIn } from '@/auth';

const getTokenUri = (code: string) => {
  const currentPolicy = getPolicy(providerCache.get() || 'microsoft-entra-id');

  const tokenUrl = new URL(process.env.AZURE_AD_TOKEN_URL as string);
  tokenUrl.searchParams.set('p', currentPolicy);
  tokenUrl.searchParams.set('client_id', process.env.AZURE_AD_CLIENT_ID as string);
  tokenUrl.searchParams.set('client_secret', process.env.AZURE_AD_CLIENT_SECRET as string);
  tokenUrl.searchParams.set('grant_type', 'authorization_code');
  tokenUrl.searchParams.set('response_type', 'token id_token');
  tokenUrl.searchParams.set('code', code);

  providerCache.delete();

  return tokenUrl.toString();
};

const getTokens = async (code: string) => {
  const response = await axios.post(getTokenUri(code), {});
  return response.data;
};

const signInWithCredentials = async (data: any) => {
  await signIn('credentials', {
    id_token: data.id_token,
    access_token: data.access_token,
    refresh_token: data.refresh_token,
    expires_on: data.expires_on,
    redirect: false
  });
};

export const GET = async (req: NextRequest) => {
  const url = new URL(req.url);
  const code = url.searchParams.get('code');

  if (!code) {
    redirect('/error?error=missing_code');
  }

  try {
    const tokens = await getTokens(code);
    await signInWithCredentials(tokens);
  } catch (err: unknown) {
    const error = err as any;

    if (isAxiosError(err)) {
      console.error('[TOKEN] Error', isAxiosError(err) ? err.toJSON() : error);
    }

    redirect(`/error?error=${error.message || 'login_failed'}`);
  }

  redirect('/uvod');
};
