import { useLocale, useTranslations } from 'next-intl';
import React, { useMemo } from 'react';
import QRCode from 'react-qr-code';
import { CurrencyCode, DataModel, encode, PaymentOptions } from 'bysquare';
import { v4 } from 'uuid';
import { Proceeding, ProceedingSource } from '@/types';
import { Dialog, ErrorImg, SecondaryButton } from '@/lib/idsk';
import { date } from '@/utils';
import { InformationDialog } from '@/components/core';

const ZOSTAVA_UHRADIT = 'ZOSTAVA_UHRADIT';

interface ProceedingPayDialogProps {
  item: Proceeding | null;
  onClose: () => void;
}

type TaxCategory = 'default' | 'trash' | 'dogs';
type PaymentType = 'default' | 'INKASO' | 'SIPO';

const translations: Record<TaxCategory, string> = {
  default: 'taxes_are_paid',
  trash: 'trash_taxes_are_paid',
  dogs: 'dog_taxes_are_paid'
};

const translationsOverpayment: Record<TaxCategory, string> = {
  default: 'taxes_overpayment',
  trash: 'trash_taxes_overpayment',
  dogs: 'dog_taxes_overpayment'
};

const getCategory = (subcategories: string[]): TaxCategory => {
  const categories = ['Poplatok za odpad', 'Daň za psa', 'Daň z nehnuteľnosti'];

  if (subcategories.includes(categories[0])) return 'trash';
  if (subcategories.includes(categories[1])) return 'dogs';
  return 'default';
};

const getPaymentMethod = (sources: ProceedingSource[]): PaymentType => {
  const method = sources.find((source) => source.key === 'PLATOBNA_METODA')?.value;
  if (method && (method === 'INKASO' || method === 'SEPA')) return 'INKASO';
  if (method && method === 'SIPO') return 'SIPO';

  return 'default';
};

const ProceedingPayDialog: React.FC<ProceedingPayDialogProps> = ({ item, onClose }) => {
  const t = useTranslations('taxes');
  const locale = useLocale();

  const isOverpayment = item && item.total < 0;
  const isPayed = item && item.total === item.paid;
  const sum = (item ? item.total - item.paid : 0).toFixed(2);

  const { paymentMethod, category }: { paymentMethod: PaymentType; category: TaxCategory } = useMemo(() => {
    if (item) {
      return { paymentMethod: getPaymentMethod(item.sources), category: getCategory(item.subcategories) };
    }

    return { paymentMethod: 'default', category: 'default' };
  }, [item]);

  const [qrCode, metadata] = useMemo(() => {
    const source = item?.sources.find((sourceItem) => sourceItem.key === ZOSTAVA_UHRADIT);

    if (item && source && !!source.metadata) {
      const data = {
        invoiceId: v4(),
        payments: [
          {
            type: PaymentOptions.PaymentOrder,
            currencyCode: CurrencyCode.EUR,
            amount: Number(sum),
            variableSymbol: source.metadata.find((meta) => meta.key === 'VS')?.value || undefined,
            specificSymbol: source.metadata.find((meta) => meta.key === 'SS')?.value || undefined,
            constantSymbol: source.metadata.find((meta) => meta.key === 'KS')?.value || undefined,
            bankAccounts: [{ iban: source.metadata.find((meta) => meta.key === 'IBAN')?.value || '' }],
            paymentDueDate:
              source.metadata.find((meta) => meta.key === 'SPLATNOST')?.value.substring(0, 10) || undefined,
            beneficiary: {
              name: source.metadata.find((meta) => meta.key === 'RECIPIENT_NAME')?.value || ''
            }
          }
        ]
      } satisfies DataModel;

      return [encode(data), source.metadata || []];
    }

    return [null, null];
  }, [item, sum]);

  if (isOverpayment) {
    return (
      <InformationDialog
        opened={!!item}
        toggleOpened={onClose}
        title={t('pay_tax')}
        heading={t(translationsOverpayment[category])}
        description={t('taxes_overpayment_description')}
      />
    );
  }

  if (isPayed) {
    return (
      <InformationDialog
        opened={!!item}
        toggleOpened={onClose}
        variant="success"
        title={t('pay_tax')}
        heading={t(translations[category])}
        description={t('no_further_actions')}
      />
    );
  }

  if (paymentMethod !== 'default') {
    return (
      <InformationDialog
        opened={!!item}
        toggleOpened={onClose}
        variant="info"
        title={t('pay_tax')}
        heading={t(`tax_${category}_payment_${paymentMethod}`)}
        description={t(`tax_payment_${paymentMethod}_desc`)}
      />
    );
  }

  return (
    <Dialog
      className="custom-dialog"
      title={t('pay_tax')}
      opened={!!item}
      toggleOpened={onClose}
      secondaryButton={<SecondaryButton onClick={onClose}>{t('close')}</SecondaryButton>}
    >
      <div className="text-center flex flex-col items-center justify-center max-w-xl">
        {qrCode ? (
          <div className="flex flex-col tb2:flex-row gap-8 items-center justify-center">
            <QRCode value={qrCode} />
            <div className="w-full text-left">
              {metadata.map((metadataItem: any) => (
                <p key={metadataItem.key}>
                  <span>{t.has(metadataItem.key) ? t(metadataItem.key) : metadataItem.key}: </span>
                  <b>{metadataItem.key === 'SPLATNOST' ? date(metadataItem.value, locale) : metadataItem.value}</b>
                </p>
              ))}
              <p className="text-2xl border-t-2 mt-1.5 pt-1 border-neutral-300">
                {t('sum')}: <b>{sum}€</b>
              </p>
            </div>
          </div>
        ) : (
          <>
            <ErrorImg className="w-48" />
            <h4>{t('no_metadata')}</h4>
          </>
        )}
      </div>
    </Dialog>
  );
};

export default ProceedingPayDialog;
